
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_textio.all;
use ieee.numeric_std.all;
use std.textio.all;

entity verification_mycpu is
end verification_mycpu;

architecture test of verification_mycpu is

    ---------------------------------------------------------------------------
    --  simulate ROM memory                                                  --
    ---------------------------------------------------------------------------
    
    type memarray_t is array (natural range <>) of integer range 0 to 255;
    type rommem_t is protected
        impure
            function read( addr : std_logic_vector ) return std_logic_vector;
            procedure write(addr : std_logic_vector; data : std_logic_vector);
    end protected rommem_t;
    type rommem_t is protected
    body
        variable rommemarray : memarray_t(0 to 262143); -- 256 kb
    impure
        function read(addr : std_logic_vector) return std_logic_vector is
        begin
            return std_logic_vector(to_unsigned(rommemarray(to_integer(unsigned(addr))), 8));
        end function read;

        procedure write(addr : std_logic_vector; data : std_logic_vector ) is
        begin
            rommemarray(to_integer(unsigned(addr))) := to_integer(unsigned(data));
        end procedure write;
    end protected body rommem_t;
    
    shared variable ROMdata : rommem_t;  -- this is the variable to work with

    procedure load_romfile(romfile_name : in string) is
        file romfi : text;
        variable i : integer := 0;
        variable str : line;
        variable data : std_logic_vector(7 downto 0);
    begin
        file_open(romfi, romfile_name, read_mode);
        while (not endfile(romfi)) and (i < 163840) loop
            readline (romfi, str);
            hread (str, data);
            ROMdata.write(std_logic_vector(to_unsigned(i, 20)), data);
            deallocate(str);
            i := i + 1;
        end loop;
    end load_romfile;


    ---------------------------------------------------------------------------
    --  simulate RAM memory                                                  --
    ---------------------------------------------------------------------------

    type mem_block;
    type mem_block_ptr is access mem_block;
    type mem_block is array(0 to 1023) of integer range 0 to 255; -- this defines a small portion (10 bit addressable) of memory
    type mem_storage_t is array(0 to 1023) of mem_block_ptr; -- this is the entire memory array type
    type memory_t is protected
        impure
            function  readr(addr : std_logic_vector(19 downto 0)) return std_logic_vector;
            procedure writer(addr : std_logic_vector(19 downto 0); data : std_logic_vector);
    end protected memory_t;
    type memory_t is protected body
            variable mem_storage : mem_storage_t;
        impure
            function readr(addr : std_logic_vector(19 downto 0)) return std_logic_vector is
                variable row : integer;
                variable col : integer;
                variable ret_int : integer;
            begin
                row := to_integer(unsigned(addr(19 downto 10)));
                col := to_integer(unsigned(addr(9 downto 0)));
                -- break down the address so you can index into the 2D array
                if mem_storage(row) = null then
                    ret_int := 165; -- no memory allocated here yet
                else
                    ret_int := mem_storage(row)(col);
                end if;
                return std_logic_vector(to_unsigned(ret_int, 8));
            end function readr;
            procedure writer(addr : std_logic_vector(19 downto 0); data : std_logic_vector ) is
                variable row : integer;
                variable col : integer;
            begin
                row := to_integer(unsigned(addr(19 downto 10)));
                col := to_integer(unsigned(addr(9 downto 0)));
                if mem_storage(row) = null then
                    mem_storage(row) := new mem_block; -- dynamically create some more ram
                    -- initialise all of the memory that were just allocated
                    for i in 0 to 1023 loop
                        mem_storage(row)(i) := 165;
                    end loop;
                end if;
                mem_storage(row)(col) := to_integer(unsigned(data));
            end procedure writer;
    end protected body memory_t;

    shared variable RAMdata : memory_t;  -- this is the variable to work with

    ---------------------------------------------------------------------------

    -- append line to logfile
    procedure log_string(str : in string) is
        file logfi : text;
        variable l : line;
    begin
        file_open(logfi, "ouput.txt", append_mode);
        write(l, str);
        writeline(logfi, l);
    end log_string;

    ---------------------------------------------------------------------------

    -- append line to hexdump file (data read)
    procedure dump_read(adr : in std_logic_vector(18 downto 0); data : in std_logic_vector(7 downto 0)) is
        file dumpfi : text;
        variable l : line;
    begin
        file_open(dumpfi, "dump.txt", append_mode);
        hwrite(l, '0' & adr);
        write(l, string'("=="));
        hwrite(l, data);
        writeline(dumpfi, l);
    end dump_read;

    -- append line to hexdump file (data write)
    procedure dump_write(adr : in std_logic_vector(18 downto 0); data : in std_logic_vector(7 downto 0)) is
        file dumpfi : text;
        variable l : line;
    begin
        file_open(dumpfi, "dump.txt", append_mode);
        hwrite(l, '0' & adr);
        write(l, string'(":="));
        hwrite(l, data);
        writeline(dumpfi, l);
    end dump_write;

    ---------------------------------------------------------------------------

  component mycpusys
    generic (
        FPGACLK   : natural := 32000000  -- FPGA clock in Hz. Must be a multiple of 4 MHz.
    );
    port (
        -- MyCPU processor signals
        CLK       : IN  std_logic;  -- input clock (24 MHz)
        N_RESET   : IN  std_logic;  -- CPU reset
        INTRES    : OUT std_logic;  -- internal (synchronous) reset
        IRQS      : IN  std_logic_vector(7 downto 1);  -- interrupt inputs of interrupt controller
        IRQ       : IN  std_logic;  -- direct CPU interrupt input
        N_HALT    : IN  std_logic;  -- CPU halt
        DATA      : INOUT std_logic_vector(7 downto 0); -- data bus
        ADDRESS   : OUT std_logic_vector(15 downto 0);  -- address bus
        N_RCOD    : OUT std_logic;  -- /RD for code memory
        N_READ    : OUT std_logic;  -- /RD for data memory
        N_WRITE   : OUT std_logic;  -- /WR for data memory
        DDIR      : OUT std_logic;  -- data bus direction: 1=CPU reads, 0=CPU writes
        N_BUSENA  : OUT std_logic;  -- enable data + address bus drivers
        BUSCLOCK  : OUT std_logic;  -- CPU bus clock
        CLK4MHZ   : OUT std_logic;  -- user clock (4 MHz)
        N_IOENA   : INOUT std_logic_vector(2 downto 0); -- I/O-enables of memory unit

        -- extended signals to connect an other FPGA/CPLD
        IORD      : OUT std_logic;  -- extended I/O: read
        IOWR      : OUT std_logic;  -- extended I/O: write
        N_LOCAL   : IN  std_logic;  -- slave FPGA acknowledges current access (master FPGA must disconnect external bus)
        N_IDERDY  : IN  std_logic;  -- IDE drive ready signal from CPLD

        --- RS232 interface signals ---
        TXD       : OUT std_logic_vector(1 downto 0);
        RXD       : IN  std_logic_vector(1 downto 0);
        N_RTS     : OUT std_logic_vector(1 downto 0);
        N_DTR     : OUT std_logic_vector(1 downto 0);
        N_CTS     : IN  std_logic_vector(1 downto 0);
        N_DSR     : IN  std_logic_vector(1 downto 0);
        N_DCD     : IN  std_logic_vector(1 downto 0);
        N_RI      : IN  std_logic_vector(1 downto 0);

        -- PS2 keyboard signals
        PS2_CLK_I : IN  std_logic;  -- keyboard clock in
        PS2_CLK_O : OUT std_logic;  -- keyboard clock out
        PS2_DAT_I : IN  std_logic;  -- keyboard data in
        PS2_DAT_O : OUT std_logic;  -- keyboard data out

        -- LCD signals
        DISP_E1   : OUT std_logic;
        DISP_E2   : OUT std_logic;
        DISP_RS   : OUT std_logic;
        DISP_RW   : OUT std_logic;
        DISP_DATA : INOUT std_logic_vector(7 downto 0);

        -- signals to memory chips
        HIGHADDR  : OUT std_logic_vector(2 downto 0); -- extended address for memory chips
        N_CSROM   : OUT std_logic;  -- select ROM memory chip
        N_CSRAM1  : OUT std_logic;  -- select RAM memory chip 1
        N_CSRAM2  : OUT std_logic;  -- select RAM memory chip 2
        N_MEMRD   : OUT std_logic;  -- read from memory chip
        N_MEMWR   : OUT std_logic;  -- write to memory chip
        
        -- signals for serial-I/O
        IO_OUT1   : OUT std_logic;  -- goes to shift register 1  (user output)
        IO_OUT2   : OUT std_logic;  -- goes to shift register 2  (status LEDs)
        IO_LOAD   : OUT std_logic;  -- fetches data in shift registers
        IO_IN     : IN  std_logic;  -- comes from shift register (user input)

        -- DIP switches and Debug
        LED       : OUT std_logic;  -- test LED and /CS-signal for CPLD
        N_SWITCHES: OUT std_logic   -- enable DIP-switch read-in
                           -- SW0: enable intctrl
                           -- SW1: enable multi-I/O unit
                           -- SW2: enable PS/2 keyboard
                           -- SW3: enable IDE controller
                           -- SW4: reserved
                           -- SW5: LCD sel.1
                           -- SW6: LCD sel.2
                           -- SW7: program mode
    );
  end component;

  component cpld is
    port (
        -- processor bus signals
        N_RESET   : IN    std_logic;
        N_READ    : IN    std_logic;
        N_WRITE   : IN    std_logic;
        N_SELECT  : IN    std_logic;
        RDY       : INOUT std_logic;
        A         : IN    std_logic_vector(5 downto 0);
        A8        : IN    std_logic;
        A9        : IN    std_logic;
        D         : INOUT std_logic_vector(7 downto 0);
        -- IDE signals
        DD        : INOUT std_logic_vector(15 downto 0);
        DA        : OUT   std_logic_vector(2 downto 0);
        N_DRESET  : OUT   std_logic;
        N_DIOW    : OUT   std_logic;
        N_DIOR    : OUT   std_logic;
        IORDY     : IN    std_logic;
        N_IOCS16  : IN    std_logic;
        N_DASP0   : IN    std_logic;
        N_DASP1   : IN    std_logic;
        N_CS10    : OUT   std_logic;
        N_CS11    : OUT   std_logic;
        N_CS20    : OUT   std_logic;
        N_CS21    : OUT   std_logic;
        -- LPT signals
        PD        : OUT   std_logic_vector(7 downto 0);
        N_STROBE  : OUT   std_logic;
        N_AUTOFD  : OUT   std_logic;
        N_PRESET  : OUT   std_logic;
        N_SELOUT  : OUT   std_logic;
        BUSY      : IN    std_logic;
        PEMPTY    : IN    std_logic;
        SELIN     : IN    std_logic;
        N_ACK     : IN    std_logic;
        N_FAULT   : IN    std_logic;
        -- others
        SCL       : OUT   std_logic;
        SDA       : INOUT std_logic;
        LED       : OUT   std_logic
    );
  end component;

--    constant CLK_PERIOD : time :=      40 ns;   -- 25 MHz clock
--    constant CLK_PERIOD : time :=   41667 ps;   -- 24 MHz clock
    constant CLK_PERIOD : time :=      30 ns;   -- 33 MHz clock
    constant BAUD_PERIOD: time := 8680556 ps;   -- 1.8432 MHz clock
    constant CFGSWITCHES: std_logic_vector(7 downto 0) := "00000011";
                           -- SW0: enable intctrl
                           -- SW1: enable multi-I/O unit
                           -- SW2: enable PS/2 keyboard
                           -- SW3: enable IDE controller
                           -- SW4: reserved
                           -- SW5: LCD sel.1
                           -- SW6: LCD sel.2
                           -- SW7: program mode

    signal CLK          : std_logic := '0';
    signal N_IDERDY     : std_logic;
    signal IORD         : std_logic;
    signal IOWR         : std_logic;
    signal N_RESET      : std_logic;
    signal INTRES       : std_logic;
    signal IRQS         : std_logic_vector(7 downto 1);
    signal IRQ          : std_logic := '0';
    signal N_HALT       : std_logic;
    signal DATA         : std_logic_vector(7 downto 0) := (others => 'Z');
    signal ADDRESS      : std_logic_vector(15 downto 0);
    signal N_RCOD       : std_logic;
    signal N_READ       : std_logic;
    signal N_WRITE      : std_logic;
    signal DDIR         : std_logic;
    signal N_BUSENA     : std_logic;
    signal BUSCLOCK     : std_logic;
    signal CLK4MHZ      : std_logic;
    signal N_IOENA      : std_logic_vector(2 downto 0);
    signal PS2_CLK_I    : std_logic := '0';
    signal PS2_CLK_O    : std_logic;
    signal PS2_DAT_I    : std_logic := '0';
    signal PS2_DAT_O    : std_logic;
    signal DISP_E1      : std_logic;
    signal DISP_E2      : std_logic;
    signal DISP_RS      : std_logic;
    signal DISP_RW      : std_logic;
    signal DISP_DATA    : std_logic_vector(7 downto 0) := (others => 'Z');
    signal HIGHADDR     : std_logic_vector(2 downto 0); -- extended address for memory chips
    signal N_CSROM      : std_logic;  -- select ROM memory chip
    signal N_CSRAM1     : std_logic;  -- select RAM memory chip 1
    signal N_CSRAM2     : std_logic;  -- select RAM memory chip 2
    signal N_MEMRD      : std_logic;  -- read from memory chip
    signal N_MEMWR      : std_logic;  -- write to memory chip
    signal LED          : std_logic;
    signal N_SWITCHES   : std_logic;
    signal TXD          : std_logic_vector(1 downto 0);
    signal RXD          : std_logic_vector(1 downto 0) := "11";
    signal N_RTS        : std_logic_vector(1 downto 0) := "00";
    signal N_DTR        : std_logic_vector(1 downto 0) := "00";
    signal N_CTS        : std_logic_vector(1 downto 0) := "00";
    signal N_DSR        : std_logic_vector(1 downto 0) := "00";
    signal N_DCD        : std_logic_vector(1 downto 0) := "00";
    signal N_RI         : std_logic_vector(1 downto 0) := "00";
    signal IO_OUT1      : std_logic;
    signal IO_OUT2      : std_logic;
    signal IO_LOAD      : std_logic;
    signal IO_IN        : std_logic := '0';
    
    -- IDE drive signals
    signal DD           : std_logic_vector(15 downto 0) := x"ABCD";
    signal DA           : std_logic_vector(2 downto 0);
    signal N_DRESET     : std_logic;
    signal N_DIOW       : std_logic;
    signal N_DIOR       : std_logic;
    signal IORDY        : std_logic := '1';
    signal N_IOCS16     : std_logic := '1';
    signal N_DASP0      : std_logic := '1';
    signal N_DASP1      : std_logic := '1';
    signal N_CS10       : std_logic;
    signal N_CS11       : std_logic;
    signal N_CS20       : std_logic;
    signal N_CS21       : std_logic;

    -- LPT signals
    signal PD           : std_logic_vector(7 downto 0);
    signal N_STROBE     : std_logic;
    signal N_AUTOFD     : std_logic;
    signal N_PRESET     : std_logic;
    signal N_SELOUT     : std_logic;
    signal BUSY         : std_logic := '0';
    signal PEMPTY       : std_logic := '1';
    signal SELIN        : std_logic := '0';
    signal N_ACK        : std_logic := '1';
    signal N_FAULT      : std_logic := '1';

    -- other signals
    signal SCL          : std_logic;
    signal SDA          : std_logic := 'H';
    signal IDE_LED      : std_logic;
    
    -- local signals for testbench
    signal iSavedData   : std_logic_vector(7 downto 0);  -- for memory write
    signal iSavedAddr   : std_logic_vector(19 downto 0); -- for memory write
    signal iLastWr      : std_logic := '1'; -- for memory write
    signal iStr         : string(1 to 200);
    signal iRxIdx       : integer := 1;
    signal iTxData      : std_logic_vector(7 downto 0);
    signal iSrLed       : std_logic_vector(7 downto 0) := (others => '0');  -- simulates a 74HC595
    signal LEDs         : std_logic_vector(7 downto 0) := (others => '0');  -- simulates a 74HC595
    signal n_intres     : std_logic;
    signal n_rdyout     : std_logic; -- from CPLD
    signal dumpEnable   : std_logic := '0';
    signal ps2data_o    : std_logic := '1';
    signal ps2clock_o   : std_logic := '1';
    signal ps2data_i    : std_logic := '1';
    signal ps2clock_i   : std_logic := '1';

begin
  
  DUT: mycpusys
  generic map (
        FPGACLK   => 33333333
--        FPGACLK   => 24000000
  )
  port map (
        CLK       => CLK,
        N_RESET   => N_RESET,
        INTRES    => INTRES,
        IRQS      => IRQS,
        IRQ       => IRQ,
        N_HALT    => N_HALT,
        DATA      => DATA,
        ADDRESS   => ADDRESS,
        N_RCOD    => N_RCOD,
        N_READ    => N_READ,
        N_WRITE   => N_WRITE,
        DDIR      => DDIR,
        N_BUSENA  => N_BUSENA,
        BUSCLOCK  => BUSCLOCK,
        CLK4MHZ   => CLK4MHZ,
        N_IOENA   => N_IOENA,
        IORD      => IORD,
        IOWR      => IOWR,
        N_LOCAL   => '1',
        N_IDERDY  => N_IDERDY,
        TXD       => TXD,
        RXD       => RXD,
        N_RTS     => N_RTS,
        N_DTR     => N_DTR,
        N_CTS     => N_CTS,
        N_DSR     => N_DSR,
        N_DCD     => N_DCD,
        N_RI      => N_RI,
        PS2_CLK_I => PS2_CLK_I,
        PS2_CLK_O => PS2_CLK_O,
        PS2_DAT_I => PS2_DAT_I,
        PS2_DAT_O => PS2_DAT_O,
        DISP_E1   => DISP_E1,
        DISP_E2   => DISP_E2,
        DISP_RS   => DISP_RS,
        DISP_RW   => DISP_RW,
        DISP_DATA => DISP_DATA,
        HIGHADDR  => HIGHADDR,
        N_CSROM   => N_CSROM,
        N_CSRAM1  => N_CSRAM1,
        N_CSRAM2  => N_CSRAM2,
        N_MEMRD   => N_MEMRD,
        N_MEMWR   => N_MEMWR,
        IO_OUT1   => IO_OUT1,
        IO_OUT2   => IO_OUT2,
        IO_LOAD   => IO_LOAD,
        IO_IN     => IO_IN,
        LED       => LED,
        N_SWITCHES=> N_SWITCHES
  );

  n_intres <= not INTRES;

  comp_CPLD:
  cpld port map (
        -- processor bus signals
        N_RESET   => n_intres,
        N_READ    => N_READ,
        N_WRITE   => N_WRITE,
        N_SELECT  => LED,
        RDY       => n_rdyout,
        A         => ADDRESS(5 downto 0),
        A8        => ADDRESS(8),
        A9        => ADDRESS(9),
        D         => DATA,
        -- IDE signals
        DD        => DD,
        DA        => DA,
        N_DRESET  => N_DRESET,
        N_DIOW    => N_DIOW,
        N_DIOR    => N_DIOR,
        IORDY     => IORDY,
        N_IOCS16  => N_IOCS16,
        N_DASP0   => N_DASP0,
        N_DASP1   => N_DASP1,
        N_CS10    => N_CS10,
        N_CS11    => N_CS11,
        N_CS20    => N_CS20,
        N_CS21    => N_CS21,
        -- LPT signals
        PD        => PD,
        N_STROBE  => N_STROBE,
        N_AUTOFD  => N_AUTOFD,
        N_PRESET  => N_PRESET,
        N_SELOUT  => N_SELOUT,
        BUSY      => BUSY,
        PEMPTY    => PEMPTY,
        SELIN     => SELIN,
        N_ACK     => N_ACK,
        N_FAULT   => N_FAULT,
        -- others
        SCL       => SCL,
        SDA       => SDA,
        LED       => IDE_LED
  );

  N_IDERDY <= '0' when n_rdyout = '0' else '1';
  
  CLK  <= not CLK     after CLK_PERIOD/2;
  
  IRQS <= "0000000";
  IRQ  <= '0';
  DD   <= x"FF7F";
  
  -- load the ROM image file
  rdfilep:
  process begin
      report "loading ROM image" severity note;
      load_romfile("../romimg.hex");
      report "ROM image loaded" severity note;
      wait;
  end process;
  
  -- stimulus
  process begin
    N_RESET <= '0';
    N_HALT <= '1';
    SDA <= '1';
 
    -- do reset
    wait for 200 ns;
    N_RESET <= '1';

    wait;
  end process;

  -- dump ROM memory reads to a file
  process (BUSCLOCK, INTRES)
  begin
    if falling_edge(BUSCLOCK) then
        if INTRES = '1' then
            dumpEnable <= '0';
        else
            if N_MEMRD = '0' then
                if dumpEnable = '1' then
                    dump_read(HIGHADDR & ADDRESS, DATA);
                end if;
            end if;
        end if;
    end if;
  end process;
  
  -- data bus simulation
  process (CLK, N_MEMRD, N_MEMWR, N_CSRAM1, N_CSRAM2, N_READ, N_CSROM, HIGHADDR, ADDRESS, DATA, iLastWr, N_BUSENA, DDIR)
  begin
    if N_MEMRD = '0' then
        if N_CSRAM1 = '0' then
            DATA <= RAMdata.readr('0' & HIGHADDR & ADDRESS);
        elsif N_CSRAM2 = '0' then
            DATA <= RAMdata.readr('1' & HIGHADDR & ADDRESS);
        elsif N_CSROM = '0' then
            DATA <= ROMdata.read(HIGHADDR & ADDRESS);
        elsif LED = '1' then -- cpld not selected
             DATA <= x"FF";
        else
            DATA <= (others => 'Z');
        end if;
    else
        DATA <= (others => 'Z');
    end if;
    if rising_edge(CLK) then
        if (iLastWr = '0') and (N_MEMWR = '1') then
            RAMdata.writer(iSavedAddr, iSavedData);
            if dumpEnable = '1' then
                dump_write(iSavedAddr(18 downto 0), iSavedData);
            end if;
        end if;
        iSavedAddr <= N_CSRAM1 & HIGHADDR & ADDRESS;
        iSavedData <= DATA;
        iLastWr    <= N_MEMWR or (N_CSRAM1 and N_CSRAM2);
    end if;
  end process;

  -- PS2 dummy
  process
  begin
    ps2clock_o <= '1';
    ps2data_o  <= '1';
    wait;
  end process;
  ps2clock_i<= not PS2_CLK_O;
  ps2data_i <= not PS2_DAT_O;
  PS2_CLK_I <= not ( ps2clock_i and ps2clock_o );
  PS2_DAT_I <= not ( ps2data_i  and ps2data_o  );
  
  -- switches, LCD data read
  process (N_SWITCHES, DISP_E1, DISP_E2, DISP_RW, DISP_RS)
  begin
    if N_SWITCHES = '0' then
        DISP_DATA <= not CFGSWITCHES;
    elsif ((DISP_E1 = '1') or (DISP_E2 = '1')) and (DISP_RW = '1') then
        DISP_DATA <= "101010" & DISP_E2 & DISP_RS;
    else
        DISP_DATA <= (others => 'Z');
    end if;
  end process;

  -- IDE disk dummy
  process (N_DIOR, N_CS10, N_CS11, N_CS20, N_CS21, DD)
  begin
    if N_DIOR = '0' then
        if N_CS10 = '0' then
            DD <= x"1234";
        elsif N_CS11 = '0' then
            DD <= x"2345";
        elsif N_CS20 = '0' then
            DD <= x"3456";
        elsif N_CS21 = '0' then
            DD <= x"4567";
        else
            DD <= x"DEAD";
        end if;
    else
        DD <= (others => 'Z');
    end if;
  end process;
  
  process
  begin
        wait until falling_edge(N_DIOW);
        wait for 30 ns;
        IORDY <= '0';
        wait for 600 ns;
        IORDY <= '1';
  end process;
  
  -- 74HC166 for digital inputs
--  process
--  begin
--    wait until rising_edge(CLK4MHZ);
--    if IO_LOAD = '0' then
--        iSrIoIn <= "11101000";
--    else
--        iSrIoIn <= iSrIoIn(6 downto 0) & '0';
--    end if;
--  end process;
--  IO_IN <= iSrIoIn(7);

  -- 74HC595 for status LEDs
  process begin
    wait until rising_edge(CLK4MHZ);
    iSrLed <= iSrLed(6 downto 0) & IO_OUT2;
  end process;
  process begin
    wait until rising_edge(IO_LOAD);
    LEDs <= iSrLed;
  end process;
  
  -- receive serial output data
  process
    variable l: line;
    variable iAscii  : integer range 0 to 255;
    variable iUnsAsc : unsigned(7 downto 0);
    variable iRXsreg : std_logic_vector(7 downto 0);
  begin
    wait until falling_edge(TXD(0));
    wait for BAUD_PERIOD / 2;
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        iRXsreg(i) := TXD(0);
        wait for BAUD_PERIOD;
    end loop;
    iUnsAsc := unsigned(iRXsreg);
    iAscii := to_integer(iUnsAsc);
    if (iAscii > 31) or (iAscii = 13) then
        iStr(iRxIdx) <= character'val(iAscii);
        if (iRxIdx = 200) or (iAscii = 13) then
            log_string(iStr(1 to iRxIdx-1));
            write(l, iStr(1 to iRxIdx-1));
            writeline(output, l);
            iRxIdx <= 1;
        else
            iRxIdx <= iRxIdx + 1;
        end if;
    end if;
  end process;

  -- test the flash memory write function
  -- (CFG-SWITCH7 needs to be set for this)
  process
  begin
    wait for 100 ns;
    wait until rising_edge(N_RESET);
    wait for 50 us;
-- wait for 20 ms;
    ---------------------------
    -- Test the debug-command interface
    
--    iTxData <= x"40";  -- "NOP"
--    RXD(0) <= '0';
--    wait for BAUD_PERIOD;
--    for i in 0 to 7 loop
--        RXD(0) <= iTxData(i);
--        wait for BAUD_PERIOD;
--    end loop;
--    RXD(0) <= '1';
--    wait for BAUD_PERIOD;

--    iTxData <= x"4F";  -- illegal command
--    RXD(0) <= '0';
--    wait for BAUD_PERIOD;
--    for i in 0 to 7 loop
--        RXD(0) <= iTxData(i);
--        wait for BAUD_PERIOD;
--    end loop;
--    RXD(0) <= '1';
--    wait for BAUD_PERIOD;
    
    iTxData <= x"42";  -- stop CPU
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

    iTxData <= x"51";  -- execute from ROM
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;
    iTxData <= x"02";
--    iTxData <= x"00";
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;
 wait;
    -- iTxData <= x"42";  -- stop CPU
    -- RXD(0) <= '0';
    -- wait for BAUD_PERIOD;
    -- for i in 0 to 7 loop
        -- RXD(0) <= iTxData(i);
        -- wait for BAUD_PERIOD;
    -- end loop;
    -- RXD(0) <= '1';
    -- wait for BAUD_PERIOD;

    -- iTxData <= x"51";  -- run CPU
    -- RXD(0) <= '0';
    -- wait for BAUD_PERIOD;
    -- for i in 0 to 7 loop
        -- RXD(0) <= iTxData(i);
        -- wait for BAUD_PERIOD;
    -- end loop;
    -- RXD(0) <= '1';
    -- wait for BAUD_PERIOD;
    -- iTxData <= x"02";
    -- RXD(0) <= '0';
    -- wait for BAUD_PERIOD;
    -- for i in 0 to 7 loop
        -- RXD(0) <= iTxData(i);
        -- wait for BAUD_PERIOD;
    -- end loop;
    -- RXD(0) <= '1';
    -- wait for BAUD_PERIOD;

    ---------------------------
    iTxData <= x"2A";  --write
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

    iTxData <= x"34";
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

    iTxData <= x"56";
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

    iTxData <= x"C3";
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

--    iTxData <= x"01";  --read
  iTxData <= x"09";  --read
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

    iTxData <= x"AB";
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

    iTxData <= x"CD";
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

    iTxData <= x"00";  --read
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

    iTxData <= x"00";
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

    iTxData <= x"80";
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

  iTxData <= x"33";  --write
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

    iTxData <= x"45";
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

    iTxData <= x"67";
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

    iTxData <= x"DE";
    RXD(0) <= '0';
    wait for BAUD_PERIOD;
    for i in 0 to 7 loop
        RXD(0) <= iTxData(i);
        wait for BAUD_PERIOD;
    end loop;
    RXD(0) <= '1';
    wait for BAUD_PERIOD;

    wait;
  end process;

end test;
