-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
--  Use Spartan3 block RAM to generate a 32kb SRAM
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;


entity ram32kb is
    port (
        res       : IN  std_logic;   -- sync. FPGA reset
        clk       : IN  std_logic;
        n_wren_i  : IN  std_logic;   -- write signal
        enable_i  : IN  std_logic;   -- memory select
        address_i : IN  std_logic_vector(14 downto 0);
        data_i    : IN  std_logic_vector(7 downto 0);
        data_o    : OUT std_logic_vector(7 downto 0)
    );
end ram32kb;


architecture rtl of ram32kb is

    ----- component RAMB16_S9 -----
    component RAMB16_S9
      generic (
         INIT : bit_vector := X"000";
         INITP_00 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INITP_01 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INITP_02 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INITP_03 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INITP_04 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INITP_05 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INITP_06 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INITP_07 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_00 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_01 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_02 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_03 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_04 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_05 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_06 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_07 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_08 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_09 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_0A : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_0B : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_0C : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_0D : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_0E : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_0F : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_10 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_11 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_12 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_13 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_14 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_15 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_16 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_17 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_18 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_19 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_1A : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_1B : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_1C : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_1D : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_1E : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_1F : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_20 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_21 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_22 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_23 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_24 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_25 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_26 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_27 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_28 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_29 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_2A : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_2B : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_2C : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_2D : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_2E : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_2F : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_30 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_31 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_32 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_33 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_34 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_35 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_36 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_37 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_38 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_39 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_3A : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_3B : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_3C : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_3D : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_3E : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         INIT_3F : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
         SRVAL : bit_vector := X"000";
         WRITE_MODE : string := "WRITE_FIRST"
      );
      port (
         DO : out STD_LOGIC_VECTOR (7 downto 0);
         DOP : out STD_LOGIC_VECTOR (0 downto 0);
         ADDR : in STD_LOGIC_VECTOR (10 downto 0);
         CLK : in STD_ULOGIC;
         DI : in STD_LOGIC_VECTOR (7 downto 0);
         DIP : in STD_LOGIC_VECTOR (0 downto 0);
         EN : in STD_ULOGIC;
         SSR : in STD_ULOGIC;
         WE : in STD_ULOGIC
      );
    end component;

    type tRAMDBUS is array (15 downto 0) of std_logic_vector(7 downto 0);
    signal ramEnable: std_logic_vector(15 downto 0);
    signal ramout   : tRAMDBUS;
    signal wrenable : std_logic;
    signal output   : std_logic_vector(7 downto 0) := (others => '0');
    signal genwren  : std_logic_vector(1 downto 0);

begin

    -- generate write-enable pulse
    proc_genwren:
    process (clk, res)
    begin
        if (res = '1') then
            genwren <= "00";
        elsif rising_edge(clk) then
            genwren(1) <= genwren(0);
            genwren(0) <= not n_wren_i;
        end if;
    end process;
    wrenable <= genwren(0) and not genwren(1);

    -- generate RAM-enable signals and data multiplexer
    mkrammux:
    process (address_i, ramout, enable_i)
    begin
        output <= ramout(0);
        for I in 0 to 15 loop
            if I = to_integer(unsigned(address_i(14 downto 11))) then
                ramEnable(I) <= enable_i;
                output <= ramout(I);
            else
                ramEnable(I) <= '0';
            end if;
        end loop;
    end process;

    -- generate data output bus
    ramdataout:
    process (clk)
    begin
        if rising_edge(clk) then
            data_o <= output;
        end if;
    end process;

    
    -- generate 16 * 2kb block RAM
    makeBlockmem: for I in 0 to 15 generate

       -- RAMB16_S9: Spartan-3/3E 2k x 8 + 1 Parity bit Single-Port RAM
       -- Xilinx HDL Language Template, version 11.1
       RAMB16_S9_inst : RAMB16_S9
       generic map (
          SRVAL => X"000", --  Ouput value upon SSR assertion
          WRITE_MODE => "WRITE_FIRST" --  WRITE_FIRST, READ_FIRST or NO_CHANGE
       )
       port map (
          DO => ramout(I),  -- 8-bit Data Output
          ADDR => address_i(10 downto 0),  -- 11-bit Address Input
          CLK => clk,    -- Clock
          DI => data_i,  -- 8-bit Data Input
          DIP => "0",    -- 1-bit parity Input
          EN => ramEnable(I), -- RAM Enable Input
          SSR => res,    -- Synchronous Set/Reset Input
          WE => wrenable -- Write Enable Input
       );
       -- End of RAMB16_S9_inst instantiation

   end generate;
   
end rtl;

