-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
--  MyCPU System
--  The system implements the following modules:
--  CPU, Interrupt Controller, Memory Unit, Multi-I/O, IDE-Controller.
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.numeric_std.to_unsigned;


entity mycpusys is
    generic (
        FPGACLK   : natural := 32000000;   -- FPGA clock in Hz. Must be a multiple of 4 MHz.
        WAITST    : integer := 1           -- number of external bus wait states
    );
    port (
        -- MyCPU processor signals
        CLK       : IN  std_logic;  -- input clock (min. 24 MHz, better 28 or 32 MHz)
        N_RESET   : IN  std_logic;  -- CPU reset
        INTRES    : OUT std_logic;  -- internal (synchronous) reset
        IRQS      : IN  std_logic_vector(7 downto 1);  -- interrupt inputs of interrupt controller
        IRQ       : IN  std_logic;  -- direct CPU interrupt input
        N_HALT    : IN  std_logic;  -- CPU halt
        DATA      : INOUT std_logic_vector(7 downto 0); -- data bus
        ADDRESS   : OUT std_logic_vector(15 downto 0);  -- address bus
        N_RCOD    : OUT std_logic;  -- /RD for code memory
        N_READ    : OUT std_logic;  -- /RD for data memory
        N_WRITE   : OUT std_logic;  -- /WR for data memory
        DDIR      : OUT std_logic;  -- data bus direction: 1=CPU reads, 0=CPU writes
        N_BUSENA  : OUT std_logic;  -- enable data + address bus drivers
        BUSCLOCK  : OUT std_logic;  -- CPU bus clock
        CLK4MHZ   : OUT std_logic;  -- user clock (4 MHz)
        N_IOENA   : INOUT std_logic_vector(2 downto 0); -- I/O-enables of memory unit

        -- extended signals to connect an other FPGA/CPLD
        IORD      : OUT std_logic;  -- extended I/O: read
        IOWR      : OUT std_logic;  -- extended I/O: write
        N_LOCAL   : IN  std_logic;  -- slave FPGA acknowledges current access (master FPGA must disconnect external bus)
        N_IDERDY  : IN  std_logic;  -- is '0' when IDE drive is ready

        --- RS232 interface signals ---
        TXD       : OUT std_logic_vector(1 downto 0);
        RXD       : IN  std_logic_vector(1 downto 0);
        N_RTS     : OUT std_logic_vector(1 downto 0);
        N_DTR     : OUT std_logic_vector(1 downto 0);
        N_CTS     : IN  std_logic_vector(1 downto 0);
        N_DSR     : IN  std_logic_vector(1 downto 0);
        N_DCD     : IN  std_logic_vector(1 downto 0);
        N_RI      : IN  std_logic_vector(1 downto 0);

        -- PS2 keyboard signals
        PS2_CLK_I : IN  std_logic;  -- keyboard clock in
        PS2_CLK_O : OUT std_logic;  -- keyboard clock out
        PS2_DAT_I : IN  std_logic;  -- keyboard data in
        PS2_DAT_O : OUT std_logic;  -- keyboard data out

        -- LCD signals
        DISP_E1   : OUT std_logic;
        DISP_E2   : OUT std_logic;
        DISP_RS   : OUT std_logic;
        DISP_RW   : OUT std_logic;
        DISP_DATA : INOUT std_logic_vector(7 downto 0);

        -- signals to memory chips
        HIGHADDR  : OUT std_logic_vector(2 downto 0); -- extended address for memory chips
        N_CSROM   : OUT std_logic;  -- select ROM memory chip
        N_CSRAM1  : OUT std_logic;  -- select RAM memory chip 1
        N_CSRAM2  : OUT std_logic;  -- select RAM memory chip 2
        N_MEMRD   : OUT std_logic;  -- read from memory chip
        N_MEMWR   : OUT std_logic;  -- write to memory chip
        
        -- signals for serial-I/O
        IO_OUT1   : OUT std_logic;  -- goes to shift register 1  (user output)
        IO_OUT2   : OUT std_logic;  -- goes to shift register 2  (status LEDs)
        IO_LOAD   : OUT std_logic;  -- fetches data in shift registers
        IO_IN     : IN  std_logic;  -- comes from shift register (user input)
        
        -- DIP switches and Debug
        LED       : OUT std_logic;  -- test LED, Chip-Select for CPLD
        N_SWITCHES: OUT std_logic   -- enable DIP-switch read-in
                           -- SW0: enable intctrl
                           -- SW1: enable multi-I/O unit
                           -- SW2: enable PS/2 keyboard
                           -- SW3: enable IDE controller
                           -- SW4: reserved
                           -- SW5: LCD sel.1
                           -- SW6: LCD sel.2
                           -- SW7: program mode
    );
end mycpusys;


architecture rtl of mycpusys is

    component processor is
    generic (
        FPGACLOCK : natural := 32000000  -- FPGA clock in Hz. Must be a multiple of 4 MHz.
    );
    port (
        CLK       : IN  std_logic;  -- input clock (24 MHz)
        N_RESET   : IN  std_logic;  -- CPU reset
        N_INTRES  : OUT std_logic;  -- internal (synchronous) reset
        IRQ       : IN  std_logic;  -- interrupt
        N_HALT    : IN  std_logic;  -- CPU halt
        DATAI     : IN  std_logic_vector(7 downto 0); -- data bus, input data
        DATAO     : OUT std_logic_vector(7 downto 0); -- data bus, output data
        ADDRESS   : OUT std_logic_vector(15 downto 0);  -- address bus
        N_RCOD    : OUT std_logic;  -- /RD for code memory
        N_READ    : OUT std_logic;  -- /RD for data memory
        N_WRITE   : OUT std_logic;  -- /WR for data memory
        DDIR      : OUT std_logic;  -- data bus direction: 1=CPU reads, 0=CPU writes
        BUSCLOCK  : OUT std_logic;  -- CPU bus clock
        CLK4MHZ   : OUT std_logic;  -- user clock (4 MHz)
        WAITSTATES: IN  std_logic_vector(2 downto 0); -- select bus waitstates 0-7
        QUITCYCLE : IN  std_logic;  -- quit current cycle (skip unneeded wait states)
        CYCSTATS  : OUT std_logic_vector(15 downto 0); -- cycle statistics (CPU "frequency")
        FETCH     : OUT std_logic   -- OP-code fetch signal for debuging
    );
    end component;

    component intctrl is
    port (
        n_reset_i : IN  std_logic;
        clk_i     : IN  std_logic;
        address_i : IN  std_logic_vector(15 downto 0);
        selected_o: OUT std_logic;
        data_i    : IN  std_logic_vector(7 downto 0);
        data_o    : OUT std_logic_vector(7 downto 0);
        dovalid_o : OUT std_logic;
        n_ioen1_i : IN  std_logic;
        n_read_i  : IN  std_logic;
        n_write_i : IN  std_logic;
        clk4MHz_i : IN  std_logic;
        irq_i     : IN  std_logic_vector(7 downto 0);
        irq_o     : OUT std_logic;
        tmrirq_o  : OUT std_logic
    );
    end component;

    component memunit is
    port (
        -- Interface to CPU
        n_reset_i : IN  std_logic;
        clk_i     : IN  std_logic;
        address_i : IN  std_logic_vector(15 downto 0);
        selected_o: OUT std_logic;
        data_i    : IN  std_logic_vector(7 downto 0);
        data_o    : OUT std_logic_vector(7 downto 0);
        dovalid_o : OUT std_logic;
        n_ioen_o  : OUT std_logic_vector(2 downto 0);
        n_rdcod_i : IN  std_logic;
        n_read_i  : IN  std_logic;
        n_write_i : IN  std_logic;
        n_wren_i  : IN  std_logic;
        cycstats_i: IN  std_logic_vector(15 downto 0);
        leds_o    : OUT std_logic_vector(3 downto 0);
        -- Interface to external memory chips
        memaddr_o : OUT std_logic_vector(20 downto 0);
        memdata_i : IN  std_logic_vector(7 downto 0);
        memdata_o : OUT std_logic_vector(7 downto 0);
        n_csram_o : OUT std_logic;
        n_csrom_o : OUT std_logic;
        n_mem_rd_o: OUT std_logic;
        n_mem_wr_o: OUT std_logic;
        -- Configuration Interface --
        doConfig_i: IN  std_logic;
        cfgExRam_i: IN  std_logic
    );
    end component;

    component miounit is
    generic (
        FPGACLOCK : natural := 32000000   -- MHz
    );
    port (
        n_reset_i : IN  std_logic;
        clk_i     : IN  std_logic;
        address_i : IN  std_logic_vector(15 downto 0);
        selected_o: OUT std_logic;
        data_i    : IN  std_logic_vector(7 downto 0);
        data_o    : OUT std_logic_vector(7 downto 0);
        dovalid_o : OUT std_logic;
        n_ioen1_i : IN  std_logic;
        n_read_i  : IN  std_logic;
        n_write_i : IN  std_logic;
        clk4MHz_i : IN  std_logic;
        irqKB_o   : OUT std_logic;
        irqU1_o   : OUT std_logic;
        irqU2_o   : OUT std_logic;
        n_halt_o  : OUT std_logic;
        --- UART1 external control ---
        u1_extc_i : IN  std_logic;       -- '1' enables external control
        u1_addr_i : IN  std_logic_vector(2 downto 0);
        u1_data_i : IN  std_logic_vector(7 downto 0);
        u1_data_o : OUT std_logic_vector(7 downto 0);
        u1_read_i : IN  std_logic;
        u1_write_i: IN  std_logic;
        --- RS232 interface signals ---
        txd_o     : OUT std_logic_vector(1 downto 0);
        rxd_i     : IN  std_logic_vector(1 downto 0);
        n_rts_o   : OUT std_logic_vector(1 downto 0);
        n_dtr_o   : OUT std_logic_vector(1 downto 0);
        n_cts_i   : IN  std_logic_vector(1 downto 0);
        n_dsr_i   : IN  std_logic_vector(1 downto 0);
        n_dcd_i   : IN  std_logic_vector(1 downto 0);
        n_ri_i    : IN  std_logic_vector(1 downto 0);
        n_out1_o  : OUT std_logic_vector(1 downto 0);
        n_out2_o  : OUT std_logic_vector(1 downto 0);
        --- PS2 keyboard signals
        kb_ena_i  : IN  std_logic;
        kb_clk_i  : IN  std_logic;
        kb_clk_o  : OUT std_logic;  -- o.c. driver enable (1 means CLK = lo)
        kb_dat_i  : IN  std_logic;
        kb_dat_o  : OUT std_logic;  -- o.c. driver enable (1 means DATA = lo)
        --- LCD signals
        lcd_ddir_o: OUT std_logic;  -- IC26 74HC245 DIR-signal
        lcd_dren_o: OUT std_logic;  -- IC26 74HC245 G-signal (low-active)
        lcd_e1_o  : OUT std_logic;
        lcd_e2_o  : OUT std_logic;
        lcd_rs_o  : OUT std_logic;
        lcd_rw_o  : OUT std_logic;
        lcd_sw_i  : IN  std_logic_vector(1 downto 0);  --switches
        lcd_dat_i : IN  std_logic_vector(7 downto 0);
        lcd_dat_o : OUT std_logic_vector(7 downto 0);
        --- digital I/O
        digi_i    : IN  std_logic_vector(7 downto 0);
        digi_o    : OUT std_logic_vector(7 downto 0)
    );
    end component;

    component idecontrol is
    generic (
        FPGACLOCK : natural := 32000000   -- MHz
    );
    port (
        n_reset_i : IN  std_logic;
        clk_i     : IN  std_logic;
        select_i  : IN  std_logic;
        n_read    : IN  std_logic;
        n_write   : IN  std_logic;
        address_i : IN  std_logic_vector(15 downto 0);
        address_o : OUT std_logic_vector(15 downto 0);
        data_i    : IN  std_logic_vector(7 downto 0);
        data_o    : OUT std_logic_vector(7 downto 0);
        n_iderdy_i: IN  std_logic;
        halt_o    : OUT std_logic
    );
    end component;

    component flashprog is
    port (
        res_i     : IN  std_logic;
        clk_i     : IN  std_logic;
        enable_i  : IN  std_logic;
        --- UART1 external control ---
        urequest_o: OUT std_logic;     -- if '1' the flashprog entity requests exclusive access to the UART
        uaddr_o   : OUT std_logic_vector(2 downto 0);
        udata_o   : OUT std_logic_vector(7 downto 0);
        udata_i   : IN  std_logic_vector(7 downto 0);
        uread_o   : OUT std_logic;
        uwrite_o  : OUT std_logic;
        --- Flash Memory Interface ---
        mrequest_o: OUT std_logic_vector(1 downto 0);  -- if not "00" the flashprog entity requests exclusive access to the memory
        maddr_o   : OUT std_logic_vector(18 downto 0);
        mdata_o   : OUT std_logic_vector(7 downto 0);
        mdata_i   : IN  std_logic_vector(7 downto 0);
        mdvalid_o : OUT std_logic;
        mread_o   : OUT std_logic;
        mwrite_o  : OUT std_logic;
        --- CPU Interface ---
        cpuHalt_o : OUT std_logic;
        cpuReset_o: OUT std_logic;
        cpuFetch_i: IN  std_logic;
        -- Configuration Interface --
        doConfig_o: OUT std_logic;
        memCfg_o  : OUT std_logic   -- 0=execute code from ROM, 1=execute code from RAM
    );
    end component;

    component ram32kb is
    port (
        res       : IN  std_logic;   -- sync. FPGA reset
        clk       : IN  std_logic;
        n_wren_i  : IN  std_logic;   -- write signal
        enable_i  : IN  std_logic;   -- memory select
        address_i : IN  std_logic_vector(14 downto 0);
        data_i    : IN  std_logic_vector(7 downto 0);
        data_o    : OUT std_logic_vector(7 downto 0)
    );
    end component;

    signal addr         : std_logic_vector(15 downto 0);
    signal ideaddr      : std_logic_vector(15 downto 0);
    signal cpu_data_in  : std_logic_vector(7 downto 0);
    signal cpu_data_out : std_logic_vector(7 downto 0);
    signal cpu_sync_dout: std_logic_vector(7 downto 0);
    signal cpu_data_dir : std_logic;
    signal dout_memint  : std_logic_vector(7 downto 0);
    signal din_memunit  : std_logic_vector(7 downto 0);
    signal mem_addr     : std_logic_vector(20 downto 0);
    signal mem_datao    : std_logic_vector(7 downto 0);
    signal datadir      : std_logic;  -- '0' means CPU writes to external bus
    signal n_res        : std_logic;
    signal res          : std_logic;
    signal cpuirq       : std_logic;
    signal irqout       : std_logic;
    signal cpufetch     : std_logic;
    signal n_rdcode     : std_logic;
    signal n_rddata     : std_logic;
    signal n_wrdata     : std_logic;
    signal databus      : std_logic_vector(7 downto 0);
    signal intperi      : std_logic; -- internal periphery selected
    signal selintern    : std_logic; -- internal periphery selected
    signal dout_irqc    : std_logic_vector(7 downto 0); -- output data of irq controller
    signal dval_irqc    : std_logic; -- data of interrupt controller valid
    signal sel_irqc     : std_logic; -- 1 if irq controller is selected
    signal dout_memu    : std_logic_vector(7 downto 0); -- output data of memory unit
    signal dval_memu    : std_logic; -- data of memory unit
    signal sel_memu     : std_logic; -- 1 if memory unit is selected
    signal dout_mio     : std_logic_vector(7 downto 0); -- output data of multi-io unit
    signal dval_mio     : std_logic; -- data of multi-io unit
    signal sel_mio      : std_logic; -- 1 if multi-io unit is selected
    signal n_rd         : std_logic; -- internal read signal
    signal n_wr         : std_logic; -- internal write signal
    signal n_lastwr     : std_logic; -- FF: last n_wrdata
    signal n_ioen       : std_logic_vector(2 downto 0);  -- IOEN from memory controller
    signal clock4       : std_logic; -- 4MHz
    signal last_clock4  : std_logic; -- 4MHz, saved from last clk
    signal memsel       : std_logic; -- flag: memory selected
    signal n_mread      : std_logic; -- memory read signal
    signal n_mwrite     : std_logic; -- memory write signal
    signal n_dlymwr     : std_logic; -- memory write signal, delayed
    signal n_mcsrom     : std_logic; -- rom memory select
    signal n_mcsram     : std_logic; -- ram memory select
    signal intirqs      : std_logic_vector(7 downto 0); -- 8 internal irqs
    signal tmrirq       : std_logic; -- timer interrupt signal
    signal kbirq        : std_logic; -- keyboard irq
    signal u1irq        : std_logic; -- uart1 irq
    signal u2irq        : std_logic; -- uart2 irq
    signal use_mio      : std_logic; -- flag from DIP-swtich: use internal multi-I/O unit
    signal use_intc     : std_logic; -- flag from DIP-swtich: use internal interrupt controller
    signal use_ide      : std_logic; -- flag from DIP-swtich: use internal IDE controller
    signal n_mio_hlt    : std_logic; -- /halt signal from multi-I/O-unit
    signal n_mio_wr     : std_logic; -- write signal for internal multi-I/O-unit
    signal n_int_wr     : std_logic; -- write signal for internal interrupt controller
    signal quitcycle    : std_logic; -- is high when current CPU cycle shall be quit to increase the speed
    signal u1_extc      : std_logic; -- flag: when set the UART1 is used for the flash program interface
    signal u1_addr      : std_logic_vector(2 downto 0);  -- UART1 address for flash programming
    signal u1_datai     : std_logic_vector(7 downto 0);  -- UART1 data input
    signal u1_datao     : std_logic_vector(7 downto 0);  -- UART1 data output
    signal u1_read      : std_logic; -- UART1 read signal for flash programming
    signal u1_write     : std_logic; -- UART1 write signal for flash programming
    signal progmode     : std_logic; -- flash programming mode
    signal mrequest     : std_logic; -- flash program interface: memory access request
    signal mchipsel     : std_logic_vector(1 downto 0);  -- flash program interface: select SRAM/Flash-chip
    signal maddr        : std_logic_vector(18 downto 0); -- flash memory address
    signal mdatao       : std_logic_vector(7 downto 0);  -- flash memory data, write
    signal mdvalid      : std_logic; -- memory data valid (data direction signal)
    signal mread        : std_logic; -- flash memory read
    signal mwrite       : std_logic; -- flash memory write
    signal kb_enable    : std_logic; -- enable keyboard support
    signal n_cpu_hlt    : std_logic; -- /halt-input for CPU
    signal lcd_ddir     : std_logic;  -- IC26 74HC245 DIR-signal
    signal lcd_dren     : std_logic;  -- IC26 74HC245 G-signal (low-active)
    signal lcd_e1       : std_logic;
    signal lcd_e2       : std_logic;
    signal lcd_rs       : std_logic;
    signal lcd_rw       : std_logic;
    signal lcd_sw       : std_logic_vector(1 downto 0);  --switches
    signal lcd_dati     : std_logic_vector(7 downto 0);
    signal lcd_dato     : std_logic_vector(7 downto 0);
    signal cfgswitch    : std_logic_vector(7 downto 0);
    signal rdswitch     : std_logic;
    signal n_selram     : std_logic;
    signal blink1       : std_logic;
    signal blink2       : std_logic;
    signal ledstatus    : std_logic;
    signal local        : std_logic;
    signal n_kbclki     : std_logic;
    signal n_kbclko     : std_logic;
    signal n_kbdati     : std_logic;
    signal n_kbdato     : std_logic;
    signal cpld_select  : std_logic;  -- chip-select for CPLD
    signal mrw_monoflop : std_logic_vector(3 downto 0);
    signal cpufreqstats : std_logic_vector(15 downto 0);
    signal wait_states  : std_logic_vector(2 downto 0);
    signal irqsd1       : std_logic_vector(7 downto 1);
    signal irqsdb       : std_logic_vector(7 downto 1);
    signal hzctr        : integer range 0 to 200000;
    signal prediv       : integer range 0 to 63;
    signal bitctr       : integer range 0 to 7;
    signal idedata      : std_logic_vector(7 downto 0);
    signal sr_leds      : std_logic_vector(7 downto 0);
    signal sr_digout    : std_logic_vector(7 downto 0);
    signal sr_digin     : std_logic_vector(7 downto 0);
    signal disp_leds    : std_logic_vector(7 downto 0);  -- 8 display leds
    signal digi_out     : std_logic_vector(7 downto 0);  -- digital outputs
    signal digi_in      : std_logic_vector(7 downto 0);  -- digital inputs
    signal n_out1       : std_logic_vector(1 downto 0);
    signal n_out2       : std_logic_vector(1 downto 0);
    signal memleds      : std_logic_vector(3 downto 0);
    signal cpu_busclk   : std_logic;
    signal bus_nread    : std_logic;
    signal bus_nwrite   : std_logic;
    signal bus_nrddly1  : std_logic;
    signal bus_nrddly2  : std_logic;
    signal bus_nwrdly   : std_logic;
    signal idehalt      : std_logic;
    signal sel_ide      : std_logic;
    signal sel_lpt      : std_logic;
    signal n_sysReset   : std_logic;
    signal sysReset     : std_logic;
    signal n_cpuReset   : std_logic;
    signal dbgCpuHalt   : std_logic;
    signal dbgCpuReset  : std_logic;
    signal dbgDoConfig  : std_logic;
    signal dbgMemConfig : std_logic;
    signal resetDlyCtr  : integer range 0 to 63 := 0;
    
begin

    -- include processor core
    myproc:
    processor generic map (
        FPGACLOCK => FPGACLK
    )
    port map (
        CLK     => CLK,
        N_RESET => n_cpuReset,
        N_INTRES=> n_res,
        IRQ     => cpuirq,
        N_HALT  => n_cpu_hlt,
        DATAI   => cpu_data_in,
        DATAO   => cpu_data_out,
        ADDRESS => addr,
        N_RCOD  => n_rdcode,
        N_READ  => n_rddata,
        N_WRITE => n_wrdata,
        DDIR    => cpu_data_dir,
        BUSCLOCK=> cpu_busclk,
        CLK4MHZ => clock4,
        WAITSTATES => wait_states,
        QUITCYCLE  => quitcycle,
        CYCSTATS=> cpufreqstats,
        FETCH   => cpufetch
    );

    -- latch CPU output data (added 2010-08-20)
    latchdbus:
    process (clk, res)
    begin
        if res = '1' then
            cpu_sync_dout <= (others => '0');
        elsif rising_edge(clk) then
            cpu_sync_dout <= cpu_data_out;
        end if;
    end process;
    
    -- include interrupt controller
    interruptctrl:
    intctrl port map (
        n_reset_i => n_res,
        clk_i     => CLK,
        address_i => addr,
        selected_o=> sel_irqc,
        data_i    => cpu_sync_dout,
        data_o    => dout_irqc,
        dovalid_o => dval_irqc,
        n_ioen1_i => n_ioen(0),
        n_read_i  => n_rd,
        n_write_i => n_int_wr,
        clk4MHz_i => clock4,
        irq_i     => intirqs,
        irq_o     => irqout,
        tmrirq_o  => tmrirq
    );

    -- include memory unit
    memory:
    memunit port map (
        -- Interface to CPU
        n_reset_i  => n_sysReset,
        clk_i      => CLK,
        address_i  => addr,
        selected_o => sel_memu,
        data_i     => cpu_sync_dout,  -- delayed cpu_data_out
        data_o     => dout_memu,
        dovalid_o  => dval_memu,
        n_ioen_o   => n_ioen,
        n_rdcod_i  => n_rdcode,
        n_read_i   => n_rddata,
        n_write_i  => n_wrdata,
        n_wren_i   => n_wr,
        cycstats_i => cpufreqstats,
        leds_o     => memleds,
        -- Interface to external memory chips
        memaddr_o  => mem_addr,
        memdata_i  => din_memunit,
        memdata_o  => open,
--        memdata_o  => mem_datao,
        n_csram_o  => n_mcsram,
        n_csrom_o  => n_mcsrom,
        n_mem_rd_o => n_mread,
        n_mem_wr_o => n_mwrite,
        -- Configuration Interface
        doConfig_i => dbgDoConfig,
        cfgExRam_i => dbgMemConfig
    );
    -- Usually component memunit loops data_i back to memdata_o,
    -- but because we do not want to have 1 cycle delay in the data path
    -- we short the data path here.
    mem_datao <= cpu_data_out;

    -- connect FPGA internal RAM
    rammem32:
    ram32kb port map (
        res        => res,
        clk        => CLK,
        n_wren_i   => n_wrdata,
        enable_i   => mem_addr(20),
        address_i  => mem_addr(14 downto 0),
        data_i     => cpu_sync_dout,
        data_o     => dout_memint
    );

    -- include multi-io unit
    multiiounit:
    miounit generic map (
        FPGACLOCK => FPGACLK
    )
    port map (
        n_reset_i => n_sysReset,
        clk_i     => CLK,
        address_i => addr,
        selected_o=> sel_mio,
        data_i    => cpu_data_out,
        data_o    => dout_mio,
        dovalid_o => dval_mio,
        n_ioen1_i => n_ioen(0),
        n_read_i  => n_rddata,
        n_write_i => n_mio_wr,
        clk4MHz_i => clock4,
        irqKB_o   => kbirq,
        irqU1_o   => u1irq,
        irqU2_o   => u2irq,
        n_halt_o  => n_mio_hlt,
        -- extra UART1 interface
        u1_extc_i => u1_extc,
        u1_addr_i => u1_addr,
        u1_data_i => u1_datai,
        u1_data_o => u1_datao,
        u1_read_i => u1_read,
        u1_write_i=> u1_write,
        --- RS323 interface signals ---
        txd_o     => TXD,
        rxd_i     => RXD,
        n_rts_o   => N_RTS,
        n_dtr_o   => N_DTR,
        n_cts_i   => N_CTS,
        n_dsr_i   => N_DSR,
        n_dcd_i   => N_DCD,
        n_ri_i    => N_RI,
        n_out1_o  => n_out1,
        n_out2_o  => n_out2,
        --- PS2 keyboard signals ---
        kb_ena_i  => kb_enable,
        kb_clk_i  => n_kbclki,
        kb_clk_o  => n_kbclko,
        kb_dat_i  => n_kbdati,
        kb_dat_o  => n_kbdato,
        --- LCD signals ---
        lcd_ddir_o=> lcd_ddir,
        lcd_dren_o=> lcd_dren,
        lcd_e1_o  => lcd_e1,
        lcd_e2_o  => lcd_e2,
        lcd_rs_o  => lcd_rs,
        lcd_rw_o  => lcd_rw,
        lcd_sw_i  => lcd_sw,
        lcd_dat_i => lcd_dati,
        lcd_dat_o => lcd_dato,
        digi_i    => digi_in,
        digi_o    => digi_out
    );

    -- IDE controller (control part)
    idectrl:
    idecontrol generic map (
        FPGACLOCK => FPGACLK
    )
    port map (
        n_reset_i => n_res,
        clk_i     => CLK,
        select_i  => sel_ide,
        n_read    => n_rddata,
        n_write   => n_wrdata,
        address_i => addr,
        address_o => ideaddr,
        data_i    => DATA,
        data_o    => idedata,
        n_iderdy_i=> N_IDERDY,
        halt_o    => idehalt
    );

    -- flash memory program interface (over RS232)
    memprgif:
    flashprog port map (
        res_i     => sysReset,
        clk_i     => CLK,
        enable_i  => progmode,
        --- UART1 external control ---
        urequest_o=> u1_extc,
        uaddr_o   => u1_addr,
        udata_o   => u1_datai,
        udata_i   => u1_datao,
        uread_o   => u1_read,
        uwrite_o  => u1_write,
        --- Flash Memory Interface ---
        mrequest_o=> mchipsel,
        maddr_o   => maddr,
        mdata_o   => mdatao,
        mdata_i   => DATA,
        mdvalid_o => mdvalid,
        mread_o   => mread,
        mwrite_o  => mwrite,
        --- CPU Interface ---
        cpuHalt_o => dbgCpuHalt,
        cpuReset_o=> dbgCpuReset,
        cpuFetch_i=> cpufetch,
        -- Configuration Interface --
        doConfig_o=> dbgDoConfig,
        memCfg_o  => dbgMemConfig
    );
    mrequest <= '0' when mchipsel = "00" else '1';

    -- set configuration (DIP switches)
    use_intc <= cfgswitch(0);
    use_mio  <= cfgswitch(1);
    use_ide  <= cfgswitch(3);
    kb_enable<= cfgswitch(2) and cfgswitch(1);
    lcd_sw   <= cfgswitch(6 downto 5);
    progmode <= cfgswitch(7);

    -- assign misc. output signals
    INTRES    <= res;
    N_RCOD    <= n_rdcode or sel_memu;
    bus_nread <= n_rddata or local or memsel or selintern;
    bus_nwrite<= n_wrdata or local or memsel or selintern;
    N_READ    <= bus_nread or bus_nrddly1;
    N_WRITE   <= bus_nwrite;
    N_BUSENA  <= mrequest or memsel or (selintern and not (n_rddata and n_rdcode));
    N_MEMWR   <= not mwrite when mrequest = '1' else n_mwrite or mem_addr(20);
    N_MEMRD   <= not mread  when mrequest = '1' else n_mread  or mem_addr(20);
    n_selram  <= '1' when mrequest = '1' else (n_mcsram or not n_mcsrom) or mem_addr(20);
    N_CSRAM1  <= '0' when mchipsel = "10" else n_selram or mem_addr(19);
    N_CSRAM2  <= '0' when mchipsel = "11" else n_selram or not mem_addr(19);
    N_CSROM   <= '0' when mchipsel = "01" else '1' when mrequest = '1' else n_mcsrom;
    N_IOENA   <= "111"  when (memsel = '1') or (selintern = '1') or (mrequest = '1') or (cpld_select = '1') else n_ioen;
    IORD      <= not (n_rddata or memsel or selintern);
    IOWR      <= not (n_wrdata or memsel or selintern);
    local     <= not N_LOCAL;
    
    -- other control signals
    memsel  <= ((not n_mcsram) or ((not n_mcsrom) and (not n_rdcode)));
    intperi <= '1' when selintern = '1' else '0';
    res     <= not n_res;
    cpuirq  <= irqout   when use_intc = '1' else u1irq or u2irq when use_mio = '1' else IRQ;
    n_mio_wr<= n_wrdata when use_mio  = '1' else '1';
    n_int_wr<= n_wr     when use_intc = '1' else '1';
    din_memunit <= dout_memint when mem_addr(20) = '1' else DATA;
    n_cpu_hlt <= N_HALT and n_mio_hlt and (not dbgCpuHalt) and (not idehalt);

    -- keyboard signals
    n_kbclki  <= not PS2_CLK_I;
    n_kbdati  <= not PS2_DAT_I;
    PS2_CLK_O <= not n_kbclko;
    PS2_DAT_O <= not n_kbdato;

    -- FPGA internal memory is very fast
    quitcycle <= '1' when (mem_addr(20) = '1') and (n_mcsram = '0') and ((n_wrdata = '0') or (n_rddata = '0')) else '0';

    -- connection to external address bus
    ADDRESS <= maddr   (15 downto 0)  when mrequest = '1' else
               mem_addr(15 downto 0)  when memsel   = '1' else
               ideaddr                when sel_ide  = '1' else
               addr;

    HIGHADDR<= maddr   (18 downto 16) when mrequest = '1' else
               mem_addr(18 downto 16);

    -- connection to external data bus
    datadir <= '0' when (intperi = '1') or ((local = '1') and (n_rddata = '0'))
                   else cpu_data_dir;  -- set write mode when intperi selected because of setup time for following ext. bus write

    cpu_data_in <= databus when intperi = '1' else
                   idedata when sel_ide = '1' else
                   DATA;

    DATA <= mdatao when (mrequest = '1') and (mdvalid = '1') else
            (others => 'Z') when (mrequest = '1') else
            cpu_data_out when (memsel = '0') and (datadir = '0') else
            mem_datao    when (memsel = '1') and ((n_mwrite = '0') or (n_dlymwr = '0')) else
            (others => 'Z');
    DDIR <= datadir;

    -- chip-select for CPLD
    sel_lpt <= '1' when (use_mio = '1') and (n_ioen(0) = '0') and (addr(10 downto 7) = "0111") else '0';
    sel_ide <= '1' when (use_ide = '1') and (n_ioen(1) = '0') and (addr(10 downto 9) = "00")   else '0';
    cpld_select <= sel_lpt or sel_ide;

    -- LCD display signals
    DISP_E1 <= lcd_e1 and not rdswitch;
    DISP_E2 <= lcd_e2 and not rdswitch;
    DISP_RS <= lcd_rs;
    DISP_RW <= lcd_rw and not rdswitch;
    DISP_DATA <= lcd_dato when (lcd_ddir = '1') and (lcd_dren = '0') and (rdswitch = '0') else (others => 'Z');
    lcd_dati<= DISP_DATA;

    -- connections to internal data bus
    databus <=  dout_irqc when dval_irqc = '1' else
                dout_memu when dval_memu = '1' else
                dout_mio  when dval_mio  = '1' else
                (others => '0');

    selintern <= (sel_irqc and use_intc) or
                 (sel_mio  and use_mio ) or
                  sel_memu;

    -- assign interrupts
    intirqs(0) <= tmrirq; 
    intirqs(1) <= kbirq when use_mio = '1' else irqsdb(1);
    intirqs(2) <= u1irq when use_mio = '1' else irqsdb(2);
    intirqs(3) <= u2irq when use_mio = '1' else irqsdb(3);
    intirqs(4) <= irqsdb(4);
    intirqs(5) <= irqsdb(5);
    intirqs(6) <= irqsdb(6);
    intirqs(7) <= irqsdb(7);
    
    -- generate system reset signal
    -- (this is for the case the FPGA is loaded and
    -- N_RESET is already high when the FPGA starts functioning)
    gensysreset:
    process (clk, N_RESET)
    begin
        if N_RESET = '0' then
            sysReset <= '1';
            resetDlyCtr <= 0;
        elsif rising_edge(clk) then
            if resetDlyCtr = 50 then
                sysReset <= '0';
            else
                resetDlyCtr <= resetDlyCtr + 1;
                sysReset <= '1';
            end if;
        end if;
    end process;
    n_sysReset <= not sysReset;
    n_cpuReset <= not (sysReset or dbgCpuReset);

    -- debounce interrupt inputs
    debirqs:
    process (clk, res)
    begin
        if res = '1' then
            irqsd1 <= (others => '0');
            irqsdb <= (others => '0');
        elsif rising_edge(clk) then
            irqsd1 <= IRQS;
            irqsdb <= irqsd1;
        end if;
    end process;

    -- set number of waitstates
    genws:
    process (clk, res)
        variable wsadd1 : integer;
    begin
        wsadd1 := WAITST + 1;
        if res = '1' then
            wait_states <= std_logic_vector(to_unsigned(WAITST, 3));
        elsif rising_edge(clk) then
            if selintern = '0' and (n_rddata = '0' or n_wrdata = '0') then
                wait_states <= std_logic_vector(to_unsigned(wsadd1, 3));
            else
                wait_states <= std_logic_vector(to_unsigned(WAITST, 3));
            end if;
        end if;
    end process;

    -- generate internal /read and /write signals
    genrw:
    process (clk, res, n_rddata, n_wrdata, n_lastwr)
    begin
        if res = '1' then
            n_lastwr <= '1';
            n_wr <= '1';
        elsif rising_edge(clk) then
            n_lastwr <= n_wrdata;
            if n_lastwr = '1' and n_wrdata = '0' then
                n_wr <= '0';
            else
                n_wr <= '1';
            end if;
        end if;
        n_rd <= n_rddata;
    end process;

    -- delay memory write signal (required for external data bus driver)
    dlymemw:
    process (clk, res)
    begin
        if res = '1' then
            n_dlymwr <= '1';
        elsif rising_edge(clk) then
            n_dlymwr <= n_mwrite;
        end if;
    end process;

    -- generate external busclock signal
    genbusclk:
    process (clk, res)
    begin
        if res = '1' then
            bus_nwrdly <= '1';
            bus_nrddly1<= '1';
            bus_nrddly2<= '1';
        elsif rising_edge(clk) then
            bus_nwrdly <= bus_nwrite;
            bus_nrddly1<= bus_nread;
            bus_nrddly2<= bus_nrddly1;
        end if;
    end process;
    BUSCLOCK <= cpu_busclk or (bus_nwrdly and bus_nrddly2);
    
    -- read configuration DIP switches
    rdswitches:
    process (clk, n_sysReset, rdswitch, DISP_DATA)
    begin
        if n_sysReset = '0' then
            rdswitch <= '1';
            cfgswitch <= not DISP_DATA;
        elsif rising_edge(clk) then
            if rdswitch = '1' then
                cfgswitch <= not DISP_DATA;
                rdswitch <= '0';
            end if;
        end if;
        N_SWITCHES <= not rdswitch;
    end process;

    -- status LED:
    --  * LED is off when CPU runs (then /CS for CPLD)
    --  * LED blinks when programming mode is selected
    --  * LED flashes when flash memory gets programmed
    process (clk, n_sysReset, hzctr)
        variable slv : std_logic_vector(17 downto 0);
    begin
        slv := std_logic_vector(to_unsigned(hzctr, 18));
        if n_sysReset = '0' then
            prediv <=  0 ;
            hzctr  <=  0 ;
            blink1 <= '0';
            blink2 <= '0';
            ledstatus <= '0';
            mrw_monoflop <= "0000";
        elsif rising_edge(clk) then
            if prediv = 63 then
                prediv <= 0;
            else
                prediv <= prediv + 1;
            end if;
            if prediv = 63 then
                if hzctr = 200000 then
                    hzctr <= 0;
                    blink1 <= not blink1;
                    blink2 <= '0';
                else
                    if hzctr = 180000 then
                        blink2 <= '1';
                    end if;
                    hzctr <= hzctr + 1;
                end if;
            end if;
            if (mread = '1') or (mwrite = '1') then
                mrw_monoflop <= "1111";
            elsif (prediv = 63) and (slv(13 downto 0) = "00000000000000") then
                mrw_monoflop <= mrw_monoflop(2 downto 0) & '0';
            end if;
            if mrw_monoflop = "0000" then
                ledstatus <= blink1;
            else
                ledstatus <= blink2;
            end if;
        end if;
    end process;
    LED <= not ledstatus when dbgCpuHalt = '1' else not cpld_select;


    -- serial interface to shift registers for port multiplication
    serio:
    process (clk, n_sysReset, sr_digout, sr_leds)
    begin
        if n_sysReset = '0' then
            IO_LOAD  <= '1';
            sr_leds  <= (others => '0');
            sr_digout<= (others => '0');
            sr_digin <= (others => '0');
            digi_in  <= (others => '0');
            bitctr   <= 0;
            last_clock4 <= '0';
            CLK4MHZ  <= '0';
        elsif rising_edge(clk) then
            if (last_clock4 = '1') and (clock4 = '0') then -- falling edge of 4MHz signal?
                if bitctr = 6 then
                    IO_LOAD <= '0';
                else
                    IO_LOAD <= '1';
                end if;
                if bitctr = 7 then
                    sr_leds   <= disp_leds;
                    sr_digout <= digi_out;
                    digi_in   <= sr_digin;
                else
                    sr_leds  (7 downto 1) <= sr_leds  (6 downto 0);
                    sr_digout(7 downto 1) <= sr_digout(6 downto 0);
                end if;
            elsif (last_clock4 = '0') and (clock4 = '1') then -- rising edge of 4MHz signal?
                sr_digin <= sr_digin(6 downto 0) & IO_IN;
                if bitctr = 7 then
                    bitctr <= 0;
                else
                    bitctr <= bitctr + 1;
                end if;
            end if;
            last_clock4 <= clock4;
            CLK4MHZ <= last_clock4;
        end if;
        IO_OUT1 <= sr_digout(7);
        IO_OUT2 <= sr_leds(7);
    end process;

    disp_leds(0) <= memleds(3);
    disp_leds(1) <= memleds(2);
    disp_leds(2) <= memleds(1);
    disp_leds(3) <= memleds(0);
    disp_leds(4) <= not n_out1(0);
    disp_leds(5) <= not n_out2(0);
    disp_leds(6) <= not n_out1(1);
    disp_leds(7) <= not n_out2(1);


end rtl;



