-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
--  Part of MyCPU processor:  Microcode Control Unit
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;


entity mcunit is
    port (
        res       : IN  std_logic;   -- FPGA reset
        n_reset_i : IN  std_logic;   -- CPU reset
        clk       : IN  std_logic;
        clk_re    : IN  std_logic;   -- clock enable: rising edge
        clk_fe    : IN  std_logic;   -- clock enable: falling edge

        -- ID-bus
        data_i    : IN  std_logic_vector(7 downto 0);

        -- Control
        irq_i     : IN  std_logic;
        n_intres_o: OUT std_logic;
        n_oe_cod_o: OUT std_logic;
        n_oe_dat_o: OUT std_logic;
        n_wr_dat_o: OUT std_logic;
        n_rdcode_o: OUT std_logic;
        n_fast_o  : OUT std_logic;
        busclk_o  : OUT std_logic;
        n_busclk_o: OUT std_logic;
        clocks_o  : OUT std_logic_vector(3 downto 0);
        enables_o : OUT std_logic_vector(3 downto 0);
        const_o   : OUT std_logic_vector(1 downto 0);
        busclock_o: OUT std_logic; -- busclk for bus connector
        fetch_o   : OUT std_logic; -- OP-code fetch signal for debuging

        -- Address-Ctrl
        n_respc_o : OUT std_logic;
        n_ldpc_o  : OUT std_logic;
        asel_o    : OUT std_logic;
        tipc_o    : OUT std_logic;  -- TIPC (one clock enable)

        -- ALU-Ctrl
        flc_i     : IN  std_logic;
        flz_i     : IN  std_logic;
        flv_i     : IN  std_logic;
        alufsel_o : OUT std_logic;
        alucsel_o : OUT std_logic;
        aluldfc_o : OUT std_logic;
        aluldvz_o : OUT std_logic;
        aluclk_o  : OUT std_logic;  -- TALU (one clock enable)
        alumode_o : OUT std_logic_vector(3 downto 0)
    );
end mcunit;


architecture rtl of mcunit is

    component ucode is
        port (
            opcode_i     : IN  std_logic_vector(7 downto 0);
            mctr_i       : IN  std_logic_vector(4 downto 0);
            irq_i        : IN  std_logic;
            cflag_i      : IN  std_logic;
            zflag_i      : IN  std_logic;
            vflag_i      : IN  std_logic;
            clocks_o     : OUT std_logic_vector(3 downto 0);
            enable_o     : OUT std_logic_vector(3 downto 0);
            alu_mode_o   : OUT std_logic_vector(3 downto 0);
            alu_fsel_o   : OUT std_logic;
            alu_csel_o   : OUT std_logic;
            n_alu_ldfc_o : OUT std_logic;
            n_alu_ldvz_o : OUT std_logic;
            const_01_o   : OUT std_logic;
            const_02_o   : OUT std_logic;
            intern_o     : OUT std_logic;
            n_loadpc_o   : OUT std_logic;
            adrsel_o     : OUT std_logic;
            n_aluclk_o   : OUT std_logic;
            n_incpc_o    : OUT std_logic;
            n_mend_o     : OUT std_logic
        );
    end component;

    signal mctr    : unsigned(4 downto 0);  -- IC6,7  74AC161
    signal opcode  : std_logic_vector(7 downto 0);  -- IC8    74AC574
    signal clocks  : std_logic_vector(3 downto 0);  -- signals to decoder 74HC138
    signal enables : std_logic_vector(3 downto 0);  -- signals to decoder 74HC138
    signal ic5ff   : std_logic_vector(7 downto 0);  -- IC5  74AC574
    signal ic4ff   : std_logic_vector(7 downto 0);  -- IC4  74AC574
    signal ic4in   : std_logic_vector(7 downto 0);  -- inputs of IC4
    signal ic12aff : std_logic;  -- IC12a  1/2 74HC74
    signal ic12bff : std_logic;  -- IC12b  1/2 74HC74
    signal ic14aff : std_logic;  -- IC14a  1/2 74AC74
    signal ic14bff : std_logic;  -- IC14b  1/2 74AC74
    signal ic15bff : std_logic;  -- IC15b  1/2 74AC74
    signal mirq    : std_logic;  -- IC15a  1/2 74AC74
    signal intern  : std_logic;  -- signal
    signal n_mend  : std_logic;  -- signal
    signal busclk  : std_logic;  -- signal
    signal busclock: std_logic;  -- signal
    signal n_busclk: std_logic;  -- signal
    signal n_intres: std_logic;  -- signal
    signal n_respc : std_logic;  -- signal
    signal n_pcclk : std_logic;  -- signal
    signal n_aluclk: std_logic;  -- signal
    signal n_oe_cod: std_logic;  -- signal

begin

    -- internal clocks
    busclk <= clk_re;
    n_busclk <= clk_fe;

    -- include code for EPROMs IC1,IC2,IC3
    ucdecoder:
    ucode port map (
        opcode_i => opcode,
        mctr_i   => ic5ff(4 downto 0),
        irq_i    => mirq,
        cflag_i  => ic5ff(5),
        zflag_i  => ic5ff(6),
        vflag_i  => ic5ff(7),
        clocks_o => clocks,
        enable_o => enables,
        alu_mode_o => ic4in(3 downto 0),
        alu_fsel_o => ic4in(4),
        alu_csel_o => ic4in(5),
        n_alu_ldfc_o => ic4in(6),
        n_alu_ldvz_o => ic4in(7),
        const_01_o => const_o(0),
        const_02_o => const_o(1),
        intern_o   => intern,
        n_loadpc_o => n_ldpc_o,
        adrsel_o   => asel_o,
        n_aluclk_o => n_aluclk,
        n_incpc_o  => n_pcclk,
        n_mend_o   => n_mend
    );
    
    ic4: -- 74AC574
    process (clk, res, ic4ff)
    begin
        if res = '1' then
            ic4ff <= (others => '0');
        elsif rising_edge(clk) then
            if n_busclk = '1' then
                ic4ff <= ic4in;
            end if;
        end if;
        alumode_o <= ic4ff(3 downto 0);
        alufsel_o <= ic4ff(4);
        alucsel_o <= ic4ff(5);
        aluldfc_o <= ic4ff(6);
        aluldvz_o <= ic4ff(7);
    end process;
    
    ic5: -- 74AC574
    process (clk, res)
    begin
        if res = '1' then
            ic5ff <= (others => '0');
        elsif rising_edge(clk) then
            if clk_re = '1' then
                ic5ff(4 downto 0) <= std_logic_vector(mctr);
                ic5ff(5) <= flc_i;
                ic5ff(6) <= flz_i;
                ic5ff(7) <= flv_i;
            end if;
        end if;
    end process;

    ic8: -- 74AC574
    process (clk, res)
    begin
        if res = '1' then
            opcode <= (others => '0');
            fetch_o <= '0';
        elsif rising_edge(clk) then
            fetch_o <= '0';
            if clk_re = '1' then
                if clocks = "0001" then  -- TOP
                    fetch_o <= '1';
                    opcode <= data_i;
                end if;
            end if;
        end if;
    end process;

    ic15: -- 74AC74
    process (clk, res)
    begin
        if res = '1' then
            mirq <= '0';
            ic15bff <= '0';
        elsif rising_edge(clk) then
            if n_intres = '0' then
                mirq <= '0';
                ic15bff <= '0';
            else
                if clk_re = '1' then
                    if clocks = "0001" then  -- TOP
                        mirq <= irq_i and ic15bff;
                    end if;
                    if clocks = "1101" then  -- TSEI
                        ic15bff <= '1';
                    end if;
                    if clocks = "0110" then  -- TCLI
                        ic15bff <= '0';
                    end if;
                end if;
            end if;
        end if;
    end process;

    ic6ic7: -- 2x 74HC161
    process (clk, res)
    begin
        if res = '1' then
            mctr <= (others => '0');
        elsif rising_edge(clk) then
            if n_intres = '0' then
                mctr <= (others => '0');
            else
                if clk_re = '1' then
                    if n_mend = '0' then
                        mctr <= (others => '0');
                    else
                        mctr <= mctr + 1;
                    end if;
                end if;
            end if;
        end if;
    end process;

    ic13: -- 74HC74
    process (clk, res)
    begin
        if res = '1' then
            n_intres <= '0';
            n_respc  <= '0';
        elsif rising_edge(clk) then
            if clk_fe = '1' then
                n_intres <= n_reset_i;
                n_respc  <= n_intres;
            end if;
        end if;
    end process;
    
    ic12: -- 74HC74
    process (clk, res)
    begin
        if res = '1' then
            ic12aff <= '1';
            ic12bff <= '1';
        elsif rising_edge(clk) then
            if n_intres = '0' then
                ic12aff <= '1';
                ic12bff <= '1';
            else
                if n_busclk = '1' then
                    ic12aff <= n_mend;
                    ic12bff <= ic12aff;
                end if;
            end if;
        end if;
    end process;
    
    ic14: -- 74AC74
    process (clk, res)
    begin
        if res = '1' then
            ic14aff <= '1';
            ic14bff <= '1';
        elsif rising_edge(clk) then
            if n_intres = '0' then
                ic14aff <= '1';
            else
                if busclk = '1' then
                    ic14aff <= ic12bff;
                end if;
            end if;
            if n_respc = '0' then
                ic14bff <= '1';
            else
                if n_busclk = '1' then
                    ic14bff <= intern;
                end if;
            end if;
        end if;
    end process;

    -- misc. outputs
    aluclk_o   <= '1' when n_aluclk = '0' and busclk = '1' else '0';
    tipc_o     <= '1' when n_pcclk  = '0' and busclk = '1' else '0';
    n_wr_dat_o <= '0' when clocks  = "0111" and busclock = '0' else '1';
    n_oe_dat_o <= '0' when enables = "0010" else '1';
    n_oe_cod   <= '0' when enables = "0001" else '1';
    n_oe_cod_o <= n_oe_cod;
    n_rdcode_o <= ic14aff and n_oe_cod;
    n_fast_o   <= ic14bff;
    n_intres_o <= n_intres;
    n_respc_o  <= n_respc;
    busclk_o   <= busclk;
    n_busclk_o <= n_busclk;
    clocks_o   <= clocks;
    enables_o  <= enables;

    -- generate bus-clock for bus connetor
    genbusclock:
    process (clk, res, busclock)
    begin
        if res = '1' then
            busclock <= '0';
        elsif rising_edge(clk) then
            if busclk = '1' then
                busclock <= '1';
            elsif n_busclk = '1' then
                busclock <= '0';
            end if;
        end if;
        busclock_o <= busclock;
    end process;

end rtl;
