-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
--  MyCPU in vhdl
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;


entity processor is
    generic (
        FPGACLOCK : natural := 32000000  -- FPGA clock in Hz. Must be a multiple of 4 MHz.
    );
    port (
        CLK       : IN  std_logic;  -- input clock (24 MHz)
        N_RESET   : IN  std_logic;  -- CPU reset
        N_INTRES  : OUT std_logic;  -- internal (synchronous) reset
        IRQ       : IN  std_logic;  -- interrupt
        N_HALT    : IN  std_logic;  -- CPU halt
        DATAI     : IN  std_logic_vector(7 downto 0); -- data bus, input data
        DATAO     : OUT std_logic_vector(7 downto 0); -- data bus, output data
        ADDRESS   : OUT std_logic_vector(15 downto 0);  -- address bus
        N_RCOD    : OUT std_logic;  -- /RD for code memory
        N_READ    : OUT std_logic;  -- /RD for data memory
        N_WRITE   : OUT std_logic;  -- /WR for data memory
        DDIR      : OUT std_logic;  -- data bus direction: 1=CPU reads, 0=CPU writes
        BUSCLOCK  : OUT std_logic;  -- CPU bus clock
        CLK4MHZ   : OUT std_logic;  -- user clock (4 MHz)
        WAITSTATES: IN  std_logic_vector(2 downto 0); -- select bus waitstates 0-7
        QUITCYCLE : IN  std_logic;  -- quit current cycle (skip unneeded wait states)
        CYCSTATS  : OUT std_logic_vector(15 downto 0); -- cycle statistics (CPU "frequency")
        FETCH     : OUT std_logic   -- OP-code fetch signal for debuging
    );
end processor;


architecture rtl of processor is

    component mcunit is
        port (
            res       : IN  std_logic;   -- FPGA reset
            n_reset_i : IN  std_logic;   -- CPU reset
            clk       : IN  std_logic;
            clk_re    : IN  std_logic;   -- clock enable: rising edge
            clk_fe    : IN  std_logic;   -- clock enable: falling edge

            -- ID-bus
            data_i    : IN  std_logic_vector(7 downto 0);

            -- Control
            irq_i     : IN  std_logic;
            n_intres_o: OUT std_logic;
            n_oe_cod_o: OUT std_logic;
            n_oe_dat_o: OUT std_logic;
            n_wr_dat_o: OUT std_logic;
            n_rdcode_o: OUT std_logic;
            n_fast_o  : OUT std_logic;
            busclk_o  : OUT std_logic;
            n_busclk_o: OUT std_logic;
            clocks_o  : OUT std_logic_vector(3 downto 0);
            enables_o : OUT std_logic_vector(3 downto 0);
            const_o   : OUT std_logic_vector(1 downto 0);
            busclock_o: OUT std_logic; -- busclk for bus connector
            fetch_o   : OUT std_logic; -- OP-code fetch signal for debuging

            -- Address-Ctrl
            n_respc_o : OUT std_logic;
            n_ldpc_o  : OUT std_logic;
            asel_o    : OUT std_logic;
            tipc_o    : OUT std_logic;  -- TIPC (one clock enable)

            -- ALU-Ctrl
            flc_i     : IN  std_logic;
            flz_i     : IN  std_logic;
            flv_i     : IN  std_logic;
            alufsel_o : OUT std_logic;
            alucsel_o : OUT std_logic;
            aluldfc_o : OUT std_logic;
            aluldvz_o : OUT std_logic;
            aluclk_o  : OUT std_logic;  -- TALU (one clock enable)
            alumode_o : OUT std_logic_vector(3 downto 0)
        );
    end component;

    component alu is
        port (
            res       : IN  std_logic;   -- FPGA reset
            clk       : IN  std_logic;
            busclk_i  : IN  std_logic;
            n_busclk_i: IN  std_logic;
            flc_o     : OUT std_logic;
            flz_o     : OUT std_logic;
            flv_o     : OUT std_logic;
            alufsel_i : IN  std_logic;
            alucsel_i : IN  std_logic;
            aluldfc_i : IN  std_logic;
            aluldvz_i : IN  std_logic;
            aluclk_i  : IN  std_logic;  -- TALU (one clock enable)
            alumode_i : IN  std_logic_vector(3 downto 0);
            clocks_i  : IN  std_logic_vector(3 downto 0);
            enables_i : IN  std_logic_vector(3 downto 0);
            data_i    : IN  std_logic_vector(7 downto 0);
            data_o    : OUT std_logic_vector(7 downto 0);
            dovalid_o : OUT std_logic   -- '1' means data_o carries valid data
        );
    end component;
    
    component prgcount is
        port (
            res       : IN  std_logic;   -- FPGA reset
            clk       : IN  std_logic;
            busclk_i  : IN  std_logic;
            n_busclk_i: IN  std_logic;
            n_respc_i : IN  std_logic;
            n_ldpc_i  : IN  std_logic;
            asel_i    : IN  std_logic;
            tipc_i    : IN  std_logic;
            clocks_i  : IN  std_logic_vector(3 downto 0);
            enables_i : IN  std_logic_vector(3 downto 0);
            data_i    : IN  std_logic_vector(7 downto 0);
            data_o    : OUT std_logic_vector(7 downto 0);
            dovalid_o : OUT std_logic;  -- '1' means data_o carries valid data
            address_o : OUT std_logic_vector(15 downto 0)
        );
    end component;

    component registers is
        port (
            res       : IN  std_logic;   -- FPGA reset
            clk       : IN  std_logic;
            busclk_i  : IN  std_logic;
            const     : IN  std_logic_vector(1 downto 0);
            clocks_i  : IN  std_logic_vector(3 downto 0);
            enables_i : IN  std_logic_vector(3 downto 0);
            data_i    : IN  std_logic_vector(7 downto 0);
            data_o    : OUT std_logic_vector(7 downto 0);
            dovalid_o : OUT std_logic  -- '1' means data_o carries valid data
        );
    end component;
    
    component timegen is
        generic (
            FPGACLK   : natural := 32000000  -- FPGA clock in Hz. Must be a multiple of 4 MHz.
        );
        port (
            n_res_i   : IN  std_logic;  -- async. reset in
            res_o     : OUT std_logic;  -- async. reset out
            n_sres_o  : OUT std_logic;  -- sync. reset out
            clk       : IN  std_logic;
            n_halt_i  : IN  std_logic;
            n_fast_i  : IN  std_logic;
            n_rdcode_i: IN  std_logic;
            n_rd_dat_i: IN  std_logic;
            n_wr_dat_i: IN  std_logic;
            waitst_i  : IN  std_logic_vector(2 downto 0);  -- number of external bus wait states
            quitcyc_i : IN  std_logic;  -- quit current cycle
            clk_re_o  : OUT std_logic;
            clk_fe_o  : OUT std_logic;
            clk4MHz_o : OUT std_logic;
            cycstat_o : OUT std_logic_vector(15 downto 0)  -- cycle statistics (CPU "frequency")
        );
    end component;

    component busif is
        port (
            n_sres_i  : IN  std_logic;  -- from timing generator
            n_intres_i: IN  std_logic;  -- from ucode control
            clk       : IN  std_logic;
            -- internal interface
            address_i : IN  std_logic_vector(15 downto 0);
            data_i    : IN  std_logic_vector(7 downto 0);
            data_o    : OUT std_logic_vector(7 downto 0);
            dovalid_o : OUT std_logic;  -- '1' means data_o carries valid data
            n_oe_cod_i: IN  std_logic;
            n_oe_dat_i: IN  std_logic;
            n_wr_dat_i: IN  std_logic;
            n_rdcode_i: IN  std_logic;
            enables_i : IN  std_logic_vector(3 downto 0);
            -- external interface
            bus_addr  : OUT std_logic_vector(15 downto 0);
            bus_datai : IN  std_logic_vector(7 downto 0);
            bus_datao : OUT std_logic_vector(7 downto 0);
            bus_rdcode: OUT std_logic;
            bus_read  : OUT std_logic;
            bus_write : OUT std_logic;
            bus_dir   : OUT std_logic  -- data bus direction: 1=CPU reads, 0=CPU writes
        );
    end component;

    signal res      : std_logic;    -- FPGA reset (not MyCPU reset)
    signal n_sres   : std_logic;    -- reset input to ucode control unit
    signal clocks   : std_logic_vector(3 downto 0);  -- T's from ucode control unit
    signal enables  : std_logic_vector(3 downto 0);  -- OE's from ucode control unit
    signal const    : std_logic_vector(1 downto 0);  -- constant value from ucode control unit
    signal n_fast   : std_logic;    -- flag: fast internal operation
    signal n_rdcode : std_logic;    -- bus access: read from code memory
    signal n_oe_cod : std_logic;    -- read code memory
    signal n_oe_dat : std_logic;    -- read data memory
    signal n_wr_dat : std_logic;    -- bus access: write to data memory
    signal clk_re   : std_logic;    -- CPU clock: rising edge enable
    signal clk_fe   : std_logic;    -- CPU clock: falling edge enable
    signal memaddr  : std_logic_vector(15 downto 0); -- memory address
    signal databus  : std_logic_vector(7 downto 0);  -- internal data bus
    signal dout_regs: std_logic_vector(7 downto 0);  -- data from registers
    signal dval_regs: std_logic;    -- flag: data output valid
    signal dout_pc  : std_logic_vector(7 downto 0);  -- data from program counter
    signal dval_pc  : std_logic;    -- flag: data output valid
    signal dout_alu : std_logic_vector(7 downto 0);  -- data from ALU
    signal dval_alu : std_logic;    -- flag: data output valid
    signal dout_bus : std_logic_vector(7 downto 0);  -- data from bus interface
    signal dval_bus : std_logic;    -- flag: data output valid
    signal busclk   : std_logic;    -- (rising edge of) busclock
    signal n_busclk : std_logic;    -- (falling edge of) busclock
    signal n_intrst : std_logic;    -- CPU internal reset
    signal n_respc  : std_logic;    -- reset program counter
    signal n_ldpc   : std_logic;    -- load program counter
    signal asel     : std_logic;    -- address select for program counter
    signal tipc     : std_logic;    -- increment program counter
    signal flag_c   : std_logic;    -- processor flag: C  (carry)
    signal flag_z   : std_logic;    -- processor flag: Z  (zero)
    signal flag_v   : std_logic;    -- processor flag: V  (sign)
    signal alufsel  : std_logic;    -- ALU flag source select
    signal alucsel  : std_logic;    -- ALU carry select
    signal aluldfc  : std_logic;    -- ALU load carry
    signal aluldvz  : std_logic;    -- ALU load V+Z flag
    signal aluclk   : std_logic;    -- ALU clock signal
    signal alumode  : std_logic_vector(3 downto 0);  -- ALU mode

begin

    core:
    mcunit port map (
        res        => res,
        n_reset_i  => n_sres,
        clk        => CLK,
        clk_re     => clk_re,
        clk_fe     => clk_fe,
        data_i     => databus,
        irq_i      => IRQ,
        n_intres_o => n_intrst,
        n_oe_cod_o => n_oe_cod,
        n_oe_dat_o => n_oe_dat,
        n_wr_dat_o => n_wr_dat,
        n_rdcode_o => n_rdcode,
        n_fast_o   => n_fast,
        busclk_o   => busclk,
        fetch_o    => FETCH,
        n_busclk_o => n_busclk,
        clocks_o   => clocks,
        enables_o  => enables,
        const_o    => const,
        busclock_o => BUSCLOCK,
        n_respc_o  => n_respc,
        n_ldpc_o   => n_ldpc,
        asel_o     => asel,
        tipc_o     => tipc,
        flc_i      => flag_c,
        flz_i      => flag_z,
        flv_i      => flag_v,
        alufsel_o  => alufsel,
        alucsel_o  => alucsel,
        aluldfc_o  => aluldfc,
        aluldvz_o  => aluldvz,
        aluclk_o   => aluclk,
        alumode_o  => alumode
    );

    busconn:
    busif port map (
        n_sres_i   => n_sres,
        n_intres_i => n_intrst,
        clk        => CLK,
        address_i  => memaddr,
        data_i     => databus,
        data_o     => dout_bus,
        dovalid_o  => dval_bus,
        n_oe_cod_i => n_oe_cod,
        n_oe_dat_i => n_oe_dat,
        n_wr_dat_i => n_wr_dat,
        n_rdcode_i => n_rdcode,
        enables_i  => enables,
        bus_addr   => ADDRESS,
        bus_datai  => DATAI,
        bus_datao  => DATAO,
        bus_rdcode => N_RCOD,
        bus_read   => N_READ,
        bus_write  => N_WRITE,
        bus_dir    => DDIR
    );

    tgen:
    timegen generic map (
        FPGACLK    => FPGACLOCK
    )
    port map (
        n_res_i    => N_RESET,
        res_o      => res,
        n_sres_o   => n_sres,
        clk        => CLK,
        n_halt_i   => N_HALT,
        n_fast_i   => n_fast,
        n_rdcode_i => n_rdcode,
        n_rd_dat_i => n_oe_dat,
        n_wr_dat_i => n_wr_dat,
        waitst_i   => WAITSTATES,
        quitcyc_i  => QUITCYCLE,
        clk_re_o   => clk_re,
        clk_fe_o   => clk_fe,
        clk4MHz_o  => CLK4MHZ,
        cycstat_o  => CYCSTATS
    );

    regs:
    registers port map (
        res        => res,
        clk        => CLK,
        busclk_i   => busclk,
        const      => const,
        clocks_i   => clocks,
        enables_i  => enables,
        data_i     => databus,
        data_o     => dout_regs,
        dovalid_o  => dval_regs
    );

    pc:
    prgcount port map (
        res        => res,
        clk        => CLK,
        busclk_i   => busclk,
        n_busclk_i => n_busclk,
        n_respc_i  => n_respc,
        n_ldpc_i   => n_ldpc,
        asel_i     => asel,
        tipc_i     => tipc,
        clocks_i   => clocks,
        enables_i  => enables,
        data_i     => databus,
        data_o     => dout_pc,
        dovalid_o  => dval_pc,
        address_o  => memaddr
    );

    alunit:
    alu port map (
        res        => res,
        clk        => CLK,
        busclk_i   => busclk,
        n_busclk_i => n_busclk,
        flc_o      => flag_c,
        flz_o      => flag_z,
        flv_o      => flag_v,
        alufsel_i  => alufsel,
        alucsel_i  => alucsel,
        aluldfc_i  => aluldfc,
        aluldvz_i  => aluldvz,
        aluclk_i   => aluclk,
        alumode_i  => alumode,
        clocks_i   => clocks,
        enables_i  => enables,
        data_i     => databus,
        data_o     => dout_alu,
        dovalid_o  => dval_alu
    );

    -- assign some signals
    N_INTRES <= n_intrst;

    -- internal databus multiplexer
    databus <=  (others => '0') when (n_sres = '0') or (n_intrst = '0') else
                dout_regs when dval_regs = '1' else
                dout_pc   when dval_pc   = '1' else
                dout_bus  when dval_bus  = '1' else
                dout_alu  when dval_alu  = '1' else
                (others => '0');

end rtl;
