-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
--  Part of MyCPU processor:  Program Counter
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;


entity prgcount is
    port (
        res       : IN  std_logic;   -- FPGA reset
        clk       : IN  std_logic;
        busclk_i  : IN  std_logic;
        n_busclk_i: IN  std_logic;
        n_respc_i : IN  std_logic;
        n_ldpc_i  : IN  std_logic;
        asel_i    : IN  std_logic;
        tipc_i    : IN  std_logic;
        clocks_i  : IN  std_logic_vector(3 downto 0);
        enables_i : IN  std_logic_vector(3 downto 0);
        data_i    : IN  std_logic_vector(7 downto 0);
        data_o    : OUT std_logic_vector(7 downto 0);
        dovalid_o : OUT std_logic;  -- '1' means data_o carries valid data
        address_o : OUT std_logic_vector(15 downto 0)
    );
end prgcount;


architecture rtl of prgcount is

    signal pctr     : unsigned(15 downto 0); -- current PC value
    signal absaddr  : std_logic_vector(15 downto 0); -- IC40,42  2x 74AC574
    signal ldadr    : std_logic_vector(15 downto 0); -- IC43,45  74HC574
    signal tadl     : std_logic;  -- signal
    signal tadh     : std_logic;  -- signal
    signal tpcl     : std_logic;  -- signal
    signal tpch     : std_logic;  -- signal
    signal oepcl    : std_logic;  -- signal
    signal oepch    : std_logic;  -- signal
    signal adpc     : std_logic;  -- FF, IC34a  1/2 74AC74
    signal n_loadpc : std_logic;  -- FF, IC34b  1/2 74AC74

begin

    -- decode T and OE signals
    tadl  <= '1' when busclk_i = '1' and clocks_i = "0010" else '0';
    tadh  <= '1' when busclk_i = '1' and clocks_i = "0011" else '0';
    tpcl  <= '1' when busclk_i = '1' and clocks_i = "0100" else '0';
    tpch  <= '1' when busclk_i = '1' and clocks_i = "0101" else '0';
    oepcl <= '1' when enables_i = "0011" else '0';
    oepch <= '1' when enables_i = "0100" else '0';

    loadadr: -- IC43,45
    process (clk, res)
    begin
        if res = '1' then
            ldadr <= (others => '0');
        elsif rising_edge(clk) then
            if tpcl = '1' then
                ldadr( 7 downto 0) <= data_i;
            end if;
            if tpch = '1' then
                ldadr(15 downto 8) <= data_i;
            end if;
        end if;
    end process;

    directadr: -- IC40,42
    process (clk, res)
    begin
        if res = '1' then
            absaddr <= (others => '0');
        elsif rising_edge(clk) then
            if tadl = '1' then
                absaddr( 7 downto 0) <= data_i;
            end if;
            if tadh = '1' then
                absaddr(15 downto 8) <= data_i;
            end if;
        end if;
    end process;

    adroutmux: -- IC36,38,40,42
    process (pctr, absaddr, adpc)
    begin
        if adpc = '0' then
            address_o <= absaddr;
        else
            address_o <= std_logic_vector(pctr);
        end if;
    end process;

    prgctr: -- IC35,37,39,41
    process (clk, res)
    begin
        if res = '1' then
            pctr <= (others => '0');
        elsif rising_edge(clk) then
            if n_respc_i = '0' then
                pctr <= (others => '0');
            else
                if tipc_i = '1' then
                    if n_loadpc = '0' then
                        pctr <= unsigned(ldadr);
                    else
                        pctr <= pctr + 1;
                    end if;
                end if;
            end if;
        end if;
    end process;
    
    pcreadback: -- IC44,46
    process (oepcl, oepch, pctr)
    begin
        if oepcl = '1' then
            data_o <= std_logic_vector(pctr( 7 downto 0));
        else
            data_o <= std_logic_vector(pctr(15 downto 8));
        end if;
        dovalid_o <= oepcl or oepch;
    end process;

    sigsync: -- IC34a,b
    process (clk, res)
    begin
        if res = '1' then
            n_loadpc <= '0';
            adpc <= '0';
        elsif rising_edge(clk) then
            if n_busclk_i = '1' then
                n_loadpc <= n_ldpc_i;
                adpc <= asel_i;
            end if;
        end if;
    end process;

end rtl;
