-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
--  Part of MyCPU processor:  Time Generator
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;


entity timegen is
    generic (
        FPGACLK   : natural := 32000000  -- FPGA clock in Hz. Must be a multiple of 4 MHz.
    );
    port (
        n_res_i   : IN  std_logic;  -- async. reset in
        res_o     : OUT std_logic;  -- async. reset out
        n_sres_o  : OUT std_logic;  -- sync. reset out
        clk       : IN  std_logic;
        n_halt_i  : IN  std_logic;
        n_fast_i  : IN  std_logic;
        n_rdcode_i: IN  std_logic;
        n_rd_dat_i: IN  std_logic;
        n_wr_dat_i: IN  std_logic;
        waitst_i  : IN  std_logic_vector(2 downto 0);  -- number of external bus wait states
        quitcyc_i : IN  std_logic;  -- quit current cycle
        clk_re_o  : OUT std_logic;
        clk_fe_o  : OUT std_logic;
        clk4MHz_o : OUT std_logic;
        cycstat_o : OUT std_logic_vector(15 downto 0)  -- cycle statistics (CPU "frequency")
    );
end timegen;


architecture rtl of timegen is

    constant fourclklo: integer := (FPGACLK+4000000)/8000000;
    constant fourclkhi: integer := FPGACLK / 8000000;
    constant fourclks : integer := fourclklo + fourclkhi;
    signal fourdivider: integer range 0 to fourclks-1;
    signal resetclocks: integer range 0 to 31;

    signal chain      : std_logic_vector(8 downto 0); -- FFs, delay chain
    signal waitstates : unsigned(2 downto 0);
    signal c0,c1,c2   : std_logic;  -- signal
    signal c3,c4,c5   : std_logic;  -- signal
    signal c6,c7,c8   : std_logic;  -- signal
    signal w0,w1,w2   : std_logic;  -- signal
    signal w3,w4,w5   : std_logic;  -- signal
    signal w6,cor     : std_logic;  -- signal
    signal reset      : std_logic;  -- signal
    signal extbus     : std_logic;  -- signal
    signal re, fe     : std_logic;  -- signal
    signal curclk     : std_logic;  -- FF, current state of the CPU clock
    signal quitcyc    : std_logic;
    signal lquitcyc   : std_logic;

    -- signals for clock statistics
    signal refclkdiv  : integer range 0 to 3999999;
    signal prediv     : integer range 0 to 15;
    signal cyccounter : unsigned(21 downto 0);
    signal cycstats   : std_logic_vector(15 downto 0);
    
begin

    -- set flag: true if external bus access
    extbus <= not (n_rd_dat_i and n_rdcode_i and n_wr_dat_i);

    -- generate correct quit-cycle-flag
    genqcyc:
    process (clk, reset)
    begin
        if reset = '1' then
            lquitcyc <= '0';
        elsif rising_edge(clk) then
            lquitcyc <= quitcyc;
        end if;
    end process;
    quitcyc <= quitcyc_i and (not curclk) and (not lquitcyc);


    -- clock generator process
    clkgen:
    process (clk, n_fast_i, extbus, n_halt_i, reset, curclk, waitstates, waitst_i,
             n_wr_dat_i, chain, w0, w1, w2, w3, w4, w5, w6, cor, re, fe, quitcyc,
             c0, c1, c2, c3, c4, c5, c6, c7, c8 )
    begin

        -- decode wait states
        -- write access lasts one clock longer
        w0 <= '0';
        w1 <= '0';
        w2 <= '0';
        w3 <= '0';
        w4 <= '0';
        w5 <= '0';
        w6 <= '0';
        if (n_wr_dat_i = '0') and (waitst_i /= "111") then
            waitstates <= unsigned(waitst_i) + 1;
        else
            waitstates <= unsigned(waitst_i);
        end if;
        case waitstates is
            when "000"   =>  w0 <= '1';
            when "001"   =>  w1 <= '1';
            when "010"   =>  w2 <= '1';
            when "011"   =>  w3 <= '1';
            when "100"   =>  w4 <= '1';
            when "101"   =>  w5 <= '1';
            when "110"   =>  w6 <= '1';
            when others =>
        end case;

        -- generate timing signals re and fe
        if reset = '1' then
            chain <= (others => '0');
        elsif rising_edge(clk) then
         chain(0) <= fe and cor and n_halt_i and (not quitcyc);
            chain(1) <= fe;
            chain(2) <= fe and chain(1);
            chain(3) <= fe and chain(2);
            chain(4) <= fe and chain(3);
            chain(5) <= fe and chain(4);
            chain(6) <= fe and chain(5);
            chain(7) <= fe and chain(6);
            chain(8) <= fe and chain(7);
        end if;
        re <= chain(0) or quitcyc;
        fe <= not chain(0);
        c0 <= (not chain(0)) and (((not n_fast_i) and (not extbus)) or quitcyc);
        c1 <= chain(1) and ((not extbus) or w0 or quitcyc);
        c2 <= chain(2) and (w1 or quitcyc);
        c3 <= chain(3) and (w2 or quitcyc);
        c4 <= chain(4) and (w3 or quitcyc);
        c5 <= chain(5) and (w4 or quitcyc);
        c6 <= chain(6) and (w5 or quitcyc);
        c7 <= chain(7) and (w6 or quitcyc);
        c8 <= chain(8);
        cor <= c0 or c1 or c2 or c3 or c4 or c5 or c6 or c7 or c8;

        -- Assign re and fe to outputs.
        -- Ensure that re and fe are only valid for one clk period.
        if reset = '1' then
            curclk <= '0';
        elsif rising_edge(clk) then
            if quitcyc = '1' then
                curclk <= '1';
            elsif fe = '1' then
                curclk <= '0';
            elsif re = '1' then
                curclk <= '1';
            end if;
        end if;
        clk_re_o <= re and not curclk;
        clk_fe_o <= fe and curclk;

    end process;


    -- generate 4MHz bus clock
    clk4mhz:
    process (clk, reset)
    begin
        if reset = '1' then
            clk4MHz_o <= '0';
            fourdivider <= 0;
        elsif rising_edge(clk) then
            if fourdivider = fourclks - 1 then
                fourdivider <= 0;
                clk4MHz_o <= '0';
            else
                if fourdivider = fourclklo - 1 then
                    clk4MHz_o <= '1';
                end if;
                fourdivider <= fourdivider + 1;
            end if;
        end if;
    end process;
 

    -- generate reset signals
    resgen:
    process (clk, n_res_i, reset)
    begin
        if n_res_i = '0' then
            reset <= '1';
            n_sres_o <= '0';
            resetclocks <= 0;
        elsif rising_edge(clk) then
            if reset = '1' then
                resetclocks <= 0;
                n_sres_o <= '0';
            else
                if resetclocks = 31 then
                    n_sres_o <= '1';
                else
                    n_sres_o <= '0';
                    resetclocks <= resetclocks + 1;
                end if;
            end if;
            reset <= '0';
        end if;
        res_o <= reset;
    end process;


    -- CPU frequency statistics
    fstats:
    process (clk, n_res_i, cycstats, cyccounter)
    begin
        if n_res_i = '0' then
            refclkdiv <= 0;
            cyccounter<= (others => '0');
            prediv    <= 0;
            cycstats  <= (others => '0');
        elsif rising_edge(clk) then
            if (re = '1') and (curclk = '0') then
                if prediv = 15 then
                    prediv <= 0;
                    cyccounter <= cyccounter + 1;
                else
                    prediv <= prediv + 1;
                end if;
            end if;
            if fourdivider = fourclklo - 1 then
                if refclkdiv = 3999999 then
                    refclkdiv <= 0;
                    -- 1 second timer triggered:
                    cycstats(15) <= not cycstats(15);
                    cycstats(14 downto 0) <= std_logic_vector(cyccounter(21 downto 7));
                    cyccounter <= (others => '0');
                    prediv <= 0;
                else
                    refclkdiv <= refclkdiv + 1;
                end if;
            end if;
        end if;
        cycstat_o <= cycstats;
    end process;

end rtl;
