-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
--  Part of MyCPU processor:  External Bus Interface
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;


entity busif is
    port (
        n_sres_i  : IN  std_logic;  -- from timing generator
        n_intres_i: IN  std_logic;  -- from ucode control
        clk       : IN  std_logic;

        -- internal interface
        address_i : IN  std_logic_vector(15 downto 0);
        data_i    : IN  std_logic_vector(7 downto 0);
        data_o    : OUT std_logic_vector(7 downto 0);
        dovalid_o : OUT std_logic;  -- '1' means data_o carries valid data
        n_oe_cod_i: IN  std_logic;
        n_oe_dat_i: IN  std_logic;
        n_wr_dat_i: IN  std_logic;
        n_rdcode_i: IN  std_logic;
        enables_i : IN  std_logic_vector(3 downto 0);

        -- external interface
        bus_addr  : OUT std_logic_vector(15 downto 0);
        bus_datai : IN  std_logic_vector(7 downto 0);
        bus_datao : OUT std_logic_vector(7 downto 0);
        bus_rdcode: OUT std_logic;
        bus_read  : OUT std_logic;
        bus_write : OUT std_logic;
        bus_dir   : OUT std_logic  -- data bus direction: 1=CPU reads, 0=CPU writes
    );
end busif;


architecture rtl of busif is

    signal dout     : std_logic_vector(7 downto 0);  -- FF
    signal reset    : std_logic; -- signal
    signal n_write  : std_logic; -- FF: write signal
    signal n_wrcyc  : std_logic_vector(1 downto 0); -- FF: write cycle flag

begin

    reset <= '1' when n_sres_i = '0' or n_intres_i = '0' else '0';

    businterface:
    process (clk, n_sres_i, reset, data_i, n_oe_cod_i, n_oe_dat_i, dout,
             n_wr_dat_i, n_rdcode_i, bus_datai, n_wrcyc, n_write, enables_i, address_i)
    begin
        if reset = '1' then
            bus_addr   <= (others => '0');
            bus_datao  <= (others => '0');
            bus_read   <= '1';
            bus_write  <= '1';
            bus_dir    <= '1';
        else
            bus_addr  <= address_i;
            bus_write <= n_wr_dat_i;
            bus_read  <= n_oe_dat_i;

            -- IC22 (74HC245)
            if (n_wrcyc(0) = '0') or (n_wr_dat_i = '0') then
                if n_wrcyc(1) = '1' then
                    bus_datao <= data_i;
                else
                    bus_datao <= dout;
                end if;
                bus_dir <= '0';
            else
                bus_datao <= dout;
                bus_dir <= '1';
            end if;
        end if;
        bus_rdcode <= n_rdcode_i;

        -- This is a trick in the MyCPU FPGA version:
        -- it ensures correct timing of write cycles
        if n_sres_i = '0' then
            dout <= (others => '0');
            n_wrcyc <= (others => '1');
        elsif rising_edge(clk) then
            n_write <= n_wr_dat_i;
            n_wrcyc(1) <= n_wrcyc(0);
            n_wrcyc(0) <= n_wr_dat_i and n_write;
            if n_wr_dat_i = '0' then
                dout <= data_i;
            end if;
        end if;

        -- assign output data
        data_o <= bus_datai;
        if (enables_i = "0001") or (enables_i = "0010") then
            dovalid_o <= '1';
        else
            dovalid_o <= '0';
        end if;

    end process;

end rtl;
