-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
--  Part of MyCPU processor:  ALU
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;


entity alu is
    port (
        res       : IN  std_logic;   -- FPGA reset
        clk       : IN  std_logic;
        busclk_i  : IN  std_logic;
        n_busclk_i: IN  std_logic;
        flc_o     : OUT std_logic;
        flz_o     : OUT std_logic;
        flv_o     : OUT std_logic;
        alufsel_i : IN  std_logic;
        alucsel_i : IN  std_logic;
        aluldfc_i : IN  std_logic;
        aluldvz_i : IN  std_logic;
        aluclk_i  : IN  std_logic;  -- TALU (one clock enable)
        alumode_i : IN  std_logic_vector(3 downto 0);
        clocks_i  : IN  std_logic_vector(3 downto 0);
        enables_i : IN  std_logic_vector(3 downto 0);
        data_i    : IN  std_logic_vector(7 downto 0);
        data_o    : OUT std_logic_vector(7 downto 0);
        dovalid_o : OUT std_logic   -- '1' means data_o carries valid data
    );
end alu;


architecture rtl of alu is

    signal areg1    : std_logic_vector(7 downto 0); -- IC60  74HC574
    signal areg2    : std_logic_vector(7 downto 0); -- IC61  74HC574
    signal flgreg   : std_logic_vector(2 downto 0); -- IC52  74HC574
    signal ic55ff   : std_logic_vector(2 downto 0); -- IC55  74HC574
    signal outreg   : std_logic_vector(7 downto 0); -- IC54  74HC574
    signal aludata  : std_logic_vector(7 downto 0); -- signal
    signal product  : std_logic_vector(15 downto 0);-- multiplication result
    signal quotient : std_logic_vector(7 downto 0); -- division result
    signal q1,q2,q3 : std_logic_vector(7 downto 0) := x"00"; -- tmp. for div
    signal q4,q5,q6 : std_logic_vector(7 downto 0) := x"00"; -- tmp. for div
    signal q7,q4l   : std_logic_vector(7 downto 0) := x"00"; -- tmp. for div
    signal tar1     : std_logic;  -- signal
    signal tar2     : std_logic;  -- signal
    signal talu     : std_logic;  -- signal
    signal ldfc     : std_logic;  -- signal
    signal ldfvz    : std_logic;  -- signal
    signal oealu    : std_logic;  -- signal
    signal oebpalu  : std_logic;  -- signal
    signal oefls    : std_logic;  -- signal
    signal flc      : std_logic;  -- IC49,  1/2 74HC74
    signal flz      : std_logic;  -- IC50a, 1/2 74HC74
    signal flv      : std_logic;  -- IC50b, 1/2 74HC74
    signal alufc    : std_logic;  -- signal
    signal alufz    : std_logic;  -- signal
    signal alufv    : std_logic;  -- signal
    signal flgc     : std_logic;  -- signal
    signal flgz     : std_logic;  -- signal
    signal flgv     : std_logic;  -- signal
    signal aflgc    : std_logic;  -- signal
    signal gen_z    : std_logic;  -- generic output for Z-Flag
    signal gen_v    : std_logic;  -- generic output for V-Flag
    signal all_1    : std_logic;  -- generic output for C-Flag
    signal m0_d     : std_logic_vector(7 downto 0); -- output data of ALU mode 0
    signal m0_c     : std_logic;  -- output C-flag of ALU mode 0
    signal m0_z     : std_logic;  -- output Z-flag of ALU mode 0
    signal m0_v     : std_logic;  -- output V-flag of ALU mode 0
    signal m1_d     : std_logic_vector(7 downto 0); -- output data of ALU mode 1
    signal m1_c     : std_logic;  -- output C-flag of ALU mode 1
    signal m1_z     : std_logic;  -- output Z-flag of ALU mode 1
    signal m1_v     : std_logic;  -- output V-flag of ALU mode 1
    signal m2_d     : std_logic_vector(7 downto 0); -- output data of ALU mode 2
    signal m2_c     : std_logic;  -- output C-flag of ALU mode 2
    signal m2_z     : std_logic;  -- output Z-flag of ALU mode 2
    signal m2_v     : std_logic;  -- output V-flag of ALU mode 2
    signal m3_d     : std_logic_vector(7 downto 0); -- output data of ALU mode 3
    signal m3_c     : std_logic;  -- output C-flag of ALU mode 3
    signal m3_z     : std_logic;  -- output Z-flag of ALU mode 3
    signal m3_v     : std_logic;  -- output V-flag of ALU mode 3
    signal m4_d     : std_logic_vector(7 downto 0); -- output data of ALU mode 4
    signal m4_c     : std_logic;  -- output C-flag of ALU mode 4
    signal m4_z     : std_logic;  -- output Z-flag of ALU mode 4
    signal m4_v     : std_logic;  -- output V-flag of ALU mode 4
    signal m5_d     : std_logic_vector(7 downto 0); -- output data of ALU mode 5
    signal m5_c     : std_logic;  -- output C-flag of ALU mode 5
    signal m5_z     : std_logic;  -- output Z-flag of ALU mode 5
    signal m5_v     : std_logic;  -- output V-flag of ALU mode 5
    signal m6_d     : std_logic_vector(7 downto 0); -- output data of ALU mode 6
    signal m6_c     : std_logic;  -- output C-flag of ALU mode 6
    signal m6_z     : std_logic;  -- output Z-flag of ALU mode 6
    signal m6_v     : std_logic;  -- output V-flag of ALU mode 6
    signal m7_d     : std_logic_vector(7 downto 0); -- output data of ALU mode 7
    signal m7_c     : std_logic;  -- output C-flag of ALU mode 7
    signal m7_z     : std_logic;  -- output Z-flag of ALU mode 7
    signal m7_v     : std_logic;  -- output V-flag of ALU mode 7
    signal m8_d     : std_logic_vector(7 downto 0); -- output data of ALU mode 8
    signal m8_c     : std_logic;  -- output C-flag of ALU mode 8
    signal m8_z     : std_logic;  -- output Z-flag of ALU mode 8
    signal m8_v     : std_logic;  -- output V-flag of ALU mode 8
    signal m9_d     : std_logic_vector(7 downto 0); -- output data of ALU mode 9
    signal m9_c     : std_logic;  -- output C-flag of ALU mode 9
    signal m9_z     : std_logic;  -- output Z-flag of ALU mode 9
    signal m9_v     : std_logic;  -- output V-flag of ALU mode 9
    signal mA_d     : std_logic_vector(7 downto 0); -- output data of ALU mode A
    signal mA_c     : std_logic;  -- output C-flag of ALU mode A
    signal mA_z     : std_logic;  -- output Z-flag of ALU mode A
    signal mA_v     : std_logic;  -- output V-flag of ALU mode A
    signal mB_d     : std_logic_vector(7 downto 0); -- output data of ALU mode B
    signal mB_c     : std_logic;  -- output C-flag of ALU mode B
    signal mB_z     : std_logic;  -- output Z-flag of ALU mode B
    signal mB_v     : std_logic;  -- output V-flag of ALU mode B
    signal mC_d     : std_logic_vector(7 downto 0); -- output data of ALU mode C
    signal mC_c     : std_logic;  -- output C-flag of ALU mode C
    signal mC_z     : std_logic;  -- output Z-flag of ALU mode C
    signal mC_v     : std_logic;  -- output V-flag of ALU mode C
    signal mD_d     : std_logic_vector(7 downto 0); -- output data of ALU mode D
    signal mD_c     : std_logic;  -- output C-flag of ALU mode D
    signal mD_z     : std_logic;  -- output Z-flag of ALU mode D
    signal mD_v     : std_logic;  -- output V-flag of ALU mode D
    signal mE_d     : std_logic_vector(7 downto 0); -- output data of ALU mode E
    signal mE_c     : std_logic;  -- output C-flag of ALU mode E
    signal mE_z     : std_logic;  -- output Z-flag of ALU mode E
    signal mE_v     : std_logic;  -- output V-flag of ALU mode E
    signal mF_d     : std_logic_vector(7 downto 0); -- output data of ALU mode F
    signal mF_c     : std_logic;  -- output C-flag of ALU mode F
    signal mF_z     : std_logic;  -- output Z-flag of ALU mode F
    signal mF_v     : std_logic;  -- output V-flag of ALU mode F
    
begin

    -- decode T and OE signals
    talu  <= aluclk_i;
    tar1  <= '1' when busclk_i = '1' and clocks_i = "1110" else '0';
    tar2  <= '1' when busclk_i = '1' and clocks_i = "1111" else '0';
    ldfc  <= '1' when n_busclk_i = '1' and aluldfc_i = '0' else '0';
    ldfvz <= '1' when n_busclk_i = '1' and aluldvz_i = '0' else '0';
    oealu <= '1' when enables_i = "1101" else '0';
    oefls <= '1' when enables_i = "1110" else '0';
    oebpalu <= '1' when enables_i = "1111" else '0';
    
    inputregs: -- IC60,61
    process (clk, res)
    begin
        if res = '1' then
            areg1 <= (others => '0');
            areg2 <= (others => '0');
        elsif rising_edge(clk) then
            if tar1 = '1' then
                areg1 <= data_i;
            end if;
            if tar2 = '1' then
                areg2 <= data_i;
            end if;
        end if;
    end process;

    aluoutputs: -- IC51,53,54
    process (clk, res, oealu, oebpalu, oefls, outreg, aludata, flv, flz, flc)
    begin
        if res = '1' then
            outreg <= (others => '0');
        elsif rising_edge(clk) then
            if talu = '1' then
                outreg <= aludata;
            end if;
        end if;
        if oealu = '1' then
            data_o <= outreg;
        elsif oebpalu = '1' then
            data_o <= aludata;
        else
            data_o <= "00000" & flv & flz & flc;
        end if;
        dovalid_o <= oealu or oebpalu or oefls;
    end process;

    aluflags: -- IC49,50,52,55
    process (clk, res, alufsel_i, ic55ff, flgreg)
    begin
        if res = '1' then
            ic55ff <= (others => '0');
            flgreg <= (others => '0');
        elsif rising_edge(clk) then
            if talu = '1' then
                ic55ff(0) <= alufc;
                ic55ff(1) <= alufz;
                ic55ff(2) <= alufv;
                flgreg <= data_i(2 downto 0);
            end if;
        end if;
        if alufsel_i = '0' then
            flgc <= ic55ff(0);
            flgz <= ic55ff(1);
            flgv <= ic55ff(2);
        else
            flgc <= flgreg(0);
            flgz <= flgreg(1);
            flgv <= flgreg(2);
        end if;
        if res = '1' then
            flc <= '0';
            flz <= '0';
            flv <= '0';
        elsif rising_edge(clk) then
            if ldfc = '1' then
                flc <= flgc;
            end if;
            if ldfvz = '1' then
                flz <= flgz;
                flv <= flgv;
            end if;
        end if;
    end process;

    -- IC48 (multiplexer with 74HC00)
    aflgc <= flgc when alucsel_i = '1' else flc;

    -- asign outputs
    flc_o <= flc;
    flz_o <= flz;
    flv_o <= flv;

    ---------------------------------------------------------------------------
    
    -- one process per ALU operation
    -- results are mux'd together at the end

    mode0_and:
    process (areg1, areg2, gen_z, gen_v)
    begin
        m0_d <= areg1 and areg2;
        m0_c <= '0';
        m0_z <= gen_z;
        m0_v <= gen_v;
    end process;

    mode1_or:
    process (areg1, areg2, gen_z, gen_v)
    begin
        m1_d <= areg1 or areg2;
        m1_c <= '0';
        m1_z <= gen_z;
        m1_v <= gen_v;
    end process;

    mode2_eor:
    process (areg1, areg2, gen_z, gen_v)
    begin
        m2_d <= areg1 xor areg2;
        m2_c <= '0';
        m2_z <= gen_z;
        m2_v <= gen_v;
    end process;

    mode3_adc:
    process (areg1, areg2, gen_z, gen_v, aflgc)
        variable s1, s2, res1, res2 : unsigned(8 downto 0);
        variable result : std_logic_vector(8 downto 0);
    begin
        s1 := '0' & unsigned(areg1);
        s2 := '0' & unsigned(areg2);
        res1 := s1 + s2;
        if aflgc = '1' then
            res2 := res1 + 1;
        else
            res2 := res1;
        end if;
        result := std_logic_vector(res2);
        m3_d <= result(7 downto 0);
        m3_c <= result(8);
        m3_z <= gen_z;
        m3_v <= gen_v;
    end process;

    mode4_sbc:
    process (areg1, areg2, gen_z, gen_v, aflgc)
        variable s1, s2, res1, res2 : unsigned(8 downto 0);
        variable result : std_logic_vector(8 downto 0);
    begin
        s1 := '1' & unsigned(areg1);
        s2 := '0' & unsigned(areg2);
        res1 := s1 - s2;
        if aflgc = '0' then
            res2 := res1 - 1;
        else
            res2 := res1;
        end if;
        result := std_logic_vector(res2);
        m4_d <= result(7 downto 0);
        m4_c <= result(8);
        m4_z <= gen_z;
        m4_v <= gen_v;
    end process;

    mode5_mull:
    process (gen_v, product)
    begin
        m5_d <= product(7 downto 0);
        m5_c <= '1';
        if product = x"0000" then
            m5_z <= '1';
        else
            m5_z <= '0';
        end if;
        m5_v <= gen_v;
    end process;

    mode6_mulh:
    process (areg1, product, m6_d)
    begin
        m6_d <= product(15 downto 8);
        if m6_d /= x"00" then
            m6_c <= '1';
        else
            m6_c <= '0';
        end if;
        if areg1 = x"00" then
            m6_z <= '1';
        else
            m6_z <= '0';
        end if;
        m6_v <= areg1(7);
    end process;

    mode7_div:
    process (areg2, gen_z, gen_v, quotient)
    begin
        if areg2 = x"00" then
            m7_d <= x"FF";
        else
            m7_d <= quotient;
        end if;
        m7_c <= '0';
        m7_z <= gen_z;
        m7_v <= gen_v;
    end process;

    mode8_cinc:
    process (areg1, gen_z, gen_v, aflgc)
        variable s : unsigned(7 downto 0);
    begin
        if aflgc = '1' then
            s := unsigned(areg1) + 1;
            m8_d <= std_logic_vector(s);
            m8_c <= gen_z;
        else
            m8_d <= areg1;
            m8_c <= '0';
        end if;
        m8_z <= gen_z;
        m8_v <= gen_v;
    end process;

    mode9_cdec:
    process (areg1, gen_z, gen_v, aflgc, all_1)
        variable s : unsigned(7 downto 0);
    begin
        if aflgc = '0' then
            s := unsigned(areg1) - 1;
            m9_d <= std_logic_vector(s);
            m9_c <= not all_1;
        else
            m9_d <= areg1;
            m9_c <= '1';
        end if;
        m9_z <= gen_z;
        m9_v <= gen_v;
    end process;

    modeA_lsl:
    process (areg1, gen_z, gen_v)
    begin
        mA_d <= areg1(6 downto 0) & '0';
        mA_c <= areg1(7);
        mA_z <= gen_z;
        mA_v <= gen_v;
    end process;

    modeB_lsr:
    process (areg1, gen_z, gen_v)
    begin
        mB_d <= '0' & areg1(7 downto 1);
        mB_c <= areg1(0);
        mB_z <= gen_z;
        mB_v <= gen_v;
    end process;

    modeC_rol:
    process (areg1, gen_z, gen_v, aflgc)
    begin
        mC_d <= areg1(6 downto 0) & aflgc;
        mC_c <= areg1(7);
        mC_z <= gen_z;
        mC_v <= gen_v;
    end process;

    modeD_ror:
    process (areg1, gen_z, gen_v, aflgc)
    begin
        mD_d <= aflgc & areg1(7 downto 1);
        mD_c <= areg1(0);
        mD_z <= gen_z;
        mD_v <= gen_v;
    end process;

    modeE_inc:
    process (areg1, gen_z, gen_v)
        variable s : unsigned(7 downto 0);
    begin
        s := unsigned(areg1) + 1;
        mE_d <= std_logic_vector(s);
        mE_c <= gen_z;
        mE_z <= gen_z;
        mE_v <= gen_v;
    end process;

    modeF_dec:
    process (areg1, gen_z, gen_v, all_1)
        variable s : unsigned(7 downto 0);
    begin
        s := unsigned(areg1) - 1;
        mF_d <= std_logic_vector(s);
        mF_c <= not all_1;
        mF_z <= gen_z;
        mF_v <= gen_v;
    end process;

    -- select ALU output according to mode
    modesel:
    process (alumode_i,
             m0_d, m0_c, m0_z, m0_v, m1_d, m1_c, m1_z, m1_v,
             m2_d, m2_c, m2_z, m2_v, m3_d, m3_c, m3_z, m3_v,
             m4_d, m4_c, m4_z, m4_v, m5_d, m5_c, m5_z, m5_v,
             m6_d, m6_c, m6_z, m6_v, m7_d, m7_c, m7_z, m7_v,
             m8_d, m8_c, m8_z, m8_v, m9_d, m9_c, m9_z, m9_v,
             mA_d, mA_c, mA_z, mA_v, mB_d, mB_c, mB_z, mB_v,
             mC_d, mC_c, mC_z, mC_v, mD_d, mD_c, mD_z, mD_v,
             mE_d, mE_c, mE_z, mE_v, mF_d, mF_c, mF_z, mF_v)
    begin
        case alumode_i is
            when  x"0"  =>  aludata <= m0_d;
                            alufc <= m0_c;
                            alufz <= m0_z;
                            alufv <= m0_v;
            when  x"1"  =>  aludata <= m1_d;
                            alufc <= m1_c;
                            alufz <= m1_z;
                            alufv <= m1_v;
            when  x"2"  =>  aludata <= m2_d;
                            alufc <= m2_c;
                            alufz <= m2_z;
                            alufv <= m2_v;
            when  x"3"  =>  aludata <= m3_d;
                            alufc <= m3_c;
                            alufz <= m3_z;
                            alufv <= m3_v;
            when  x"4"  =>  aludata <= m4_d;
                            alufc <= m4_c;
                            alufz <= m4_z;
                            alufv <= m4_v;
            when  x"5"  =>  aludata <= m5_d;
                            alufc <= m5_c;
                            alufz <= m5_z;
                            alufv <= m5_v;
            when  x"6"  =>  aludata <= m6_d;
                            alufc <= m6_c;
                            alufz <= m6_z;
                            alufv <= m6_v;
            when  x"7"  =>  aludata <= m7_d;
                            alufc <= m7_c;
                            alufz <= m7_z;
                            alufv <= m7_v;
            when  x"8"  =>  aludata <= m8_d;
                            alufc <= m8_c;
                            alufz <= m8_z;
                            alufv <= m8_v;
            when  x"9"  =>  aludata <= m9_d;
                            alufc <= m9_c;
                            alufz <= m9_z;
                            alufv <= m9_v;
            when  x"A"  =>  aludata <= mA_d;
                            alufc <= mA_c;
                            alufz <= mA_z;
                            alufv <= mA_v;
            when  x"B"  =>  aludata <= mB_d;
                            alufc <= mB_c;
                            alufz <= mB_z;
                            alufv <= mB_v;
            when  x"C"  =>  aludata <= mC_d;
                            alufc <= mC_c;
                            alufz <= mC_z;
                            alufv <= mC_v;
            when  x"D"  =>  aludata <= mD_d;
                            alufc <= mD_c;
                            alufz <= mD_z;
                            alufv <= mD_v;
            when  x"E"  =>  aludata <= mE_d;
                            alufc <= mE_c;
                            alufz <= mE_z;
                            alufv <= mE_v;
            when others =>  aludata <= mF_d;
                            alufc <= mF_c;
                            alufz <= mF_z;
                            alufv <= mF_v;
        end case;
    end process;

    -- generate generic output flags
    all_1 <= '1' when aludata = x"FF"  else '0';
    gen_z <= '1' when aludata = x"00"  else '0';
    gen_v <= '1' when aludata(7) = '1' else '0';

    -- generate a 8x8=16 multiplier
    multiplier:
    process (areg1, areg2)
        variable m1, m2 : unsigned(7 downto 0);
        variable p : unsigned(15 downto 0);
    begin
        m1 := unsigned(areg1);
        m2 := unsigned(areg2);
        p  := m1 * m2;
        product <= std_logic_vector(p);
    end process;

    -- generate a 8/8=8 divider:
    -- "quotient <= areg1 / areg2;"
    -- This divider is slow because there are
    -- 4+4 adders and 4+4 comparators in a chain.
    divider:
    process (clk, areg1, areg2, q1, q2, q3, q4, q5, q6, q7, q4l)
        variable c7,c6,c5,c4,c3,c2,c1,c0 : std_logic_vector(7 downto 0);
        variable z7,z6,z5,z4,z3,z2,z1,z0 : unsigned(7 downto 0);
    begin
        if ("0000000" & areg1(7)) < areg2 then
            q7 <= areg1;
            quotient(7) <= '0';
        else
            c7 := areg2(0) & "0000000";
            z7 := unsigned(areg1) - unsigned(c7);
            q7 <= std_logic_vector(z7);
            quotient(7) <= '1';
        end if;
        if ("000000" & q7(7 downto 6)) < areg2 then
            q6 <= q7;
            quotient(6) <= '0';
        else
            c6 := areg2(1 downto 0) & "000000";
            z6 := unsigned(q7) - unsigned(c6);
            q6 <= std_logic_vector(z6);
            quotient(6) <= '1';
        end if;
        if ("00000" & q6(7 downto 5)) < areg2 then
            q5 <= q6;
            quotient(5) <= '0';
        else
            c5 := areg2(2 downto 0) & "00000";
            z5 := unsigned(q6) - unsigned(c5);
            q5 <= std_logic_vector(z5);
            quotient(5) <= '1';
        end if;
        
       -- For a better timing performance the divider chain
       -- is broken up into two parts. This is possible 
       -- because one CPU cycle lasts at least two FPGA cycles.
       -- <begin>
       if ("0000" & q5(7 downto 4)) < areg2 then
            q4l <= q5;
            quotient(4) <= '0';
        else
            c4 := areg2(3 downto 0) & "0000";
            z4 := unsigned(q5) - unsigned(c4);
            q4l <= std_logic_vector(z4);
            quotient(4) <= '1';
        end if;
        if rising_edge(clk) then
            q4 <= q4l;
        end if;
        -- <end>

        if ("000" & q4(7 downto 3)) < areg2 then
            q3 <= q4;
            quotient(3) <= '0';
        else
            c3 := areg2(4 downto 0) & "000";
            z3 := unsigned(q4) - unsigned(c3);
            q3 <= std_logic_vector(z3);
            quotient(3) <= '1';
        end if;
        if ("00" & q3(7 downto 2)) < areg2 then
            q2 <= q3;
            quotient(2) <= '0';
        else
            c2 := areg2(5 downto 0) & "00";
            z2 := unsigned(q3) - unsigned(c2);
            q2 <= std_logic_vector(z2);
            quotient(2) <= '1';
        end if;
        if ('0' & q2(7 downto 1)) < areg2 then
            q1 <= q2;
            quotient(1) <= '0';
        else
            c1 := areg2(6 downto 0) & '0';
            z1 := unsigned(q2) - unsigned(c1);
            q1 <= std_logic_vector(z1);
            quotient(1) <= '1';
        end if;
        if q1 < areg2 then
            quotient(0) <= '0';
        else
            quotient(0) <= '1';
        end if;
    end process;

end rtl;
