-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
--  Component:  Multi-I/O-Unit
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;


entity miounit is
    generic (
        FPGACLOCK : natural := 32000000   -- MHz
    );
    port (
        n_reset_i : IN  std_logic;
        clk_i     : IN  std_logic;
        address_i : IN  std_logic_vector(15 downto 0);
        selected_o: OUT std_logic;
        data_i    : IN  std_logic_vector(7 downto 0);
        data_o    : OUT std_logic_vector(7 downto 0);
        dovalid_o : OUT std_logic;
        n_ioen1_i : IN  std_logic;
        n_read_i  : IN  std_logic;
        n_write_i : IN  std_logic;
        clk4MHz_i : IN  std_logic;
        irqKB_o   : OUT std_logic;
        irqU1_o   : OUT std_logic;
        irqU2_o   : OUT std_logic;
        n_halt_o  : OUT std_logic;
        --- UART1 external control ---
        u1_extc_i : IN  std_logic;       -- '1' enables external control
        u1_addr_i : IN  std_logic_vector(2 downto 0);
        u1_data_i : IN  std_logic_vector(7 downto 0);
        u1_data_o : OUT std_logic_vector(7 downto 0);
        u1_read_i : IN  std_logic;
        u1_write_i: IN  std_logic;
        --- RS232 interface signals ---
        txd_o     : OUT std_logic_vector(1 downto 0);
        rxd_i     : IN  std_logic_vector(1 downto 0);
        n_rts_o   : OUT std_logic_vector(1 downto 0);
        n_dtr_o   : OUT std_logic_vector(1 downto 0);
        n_cts_i   : IN  std_logic_vector(1 downto 0);
        n_dsr_i   : IN  std_logic_vector(1 downto 0);
        n_dcd_i   : IN  std_logic_vector(1 downto 0);
        n_ri_i    : IN  std_logic_vector(1 downto 0);
        n_out1_o  : OUT std_logic_vector(1 downto 0);
        n_out2_o  : OUT std_logic_vector(1 downto 0);
        --- PS2 keyboard signals
        kb_ena_i  : IN  std_logic;
        kb_clk_i  : IN  std_logic;
        kb_clk_o  : OUT std_logic;  -- o.c. driver enable (1 means CLK = lo)
        kb_dat_i  : IN  std_logic;
        kb_dat_o  : OUT std_logic;  -- o.c. driver enable (1 means DATA = lo)
        --- LCD signals
        lcd_ddir_o: OUT std_logic;  -- IC26 74HC245 DIR-signal
        lcd_dren_o: OUT std_logic;  -- IC26 74HC245 G-signal (low-active)
        lcd_e1_o  : OUT std_logic;
        lcd_e2_o  : OUT std_logic;
        lcd_rs_o  : OUT std_logic;
        lcd_rw_o  : OUT std_logic;
        lcd_sw_i  : IN  std_logic_vector(1 downto 0);  --switches
        lcd_dat_i : IN  std_logic_vector(7 downto 0);
        lcd_dat_o : OUT std_logic_vector(7 downto 0);
        --- digital I/O
        digi_i    : IN  std_logic_vector(7 downto 0);
        digi_o    : OUT std_logic_vector(7 downto 0)
    );
end miounit;


architecture rtl of miounit is

    component uart_16750 is
    port (
        CLK         : in std_logic;                             -- Clock
        RST         : in std_logic;                             -- Reset
        BAUDCE      : in std_logic;                             -- Baudrate generator clock enable
        CS          : in std_logic;                             -- Chip select
        WR          : in std_logic;                             -- Write to UART
        RD          : in std_logic;                             -- Read from UART
        A           : in std_logic_vector(2 downto 0);          -- Register select
        DIN         : in std_logic_vector(7 downto 0);          -- Data bus input
        DOUT        : out std_logic_vector(7 downto 0);         -- Data bus output
        DDIS        : out std_logic;                            -- Driver disable
        INT         : out std_logic;                            -- Interrupt output
        OUT1N       : out std_logic;                            -- Output 1
        OUT2N       : out std_logic;                            -- Output 2
        RCLK        : in std_logic;                             -- Receiver clock (16x baudrate)
        BAUDOUTN    : out std_logic;                            -- Baudrate generator output (16x baudrate)
        RTSN        : out std_logic;                            -- RTS output
        DTRN        : out std_logic;                            -- DTR output
        CTSN        : in std_logic;                             -- CTS input
        DSRN        : in std_logic;                             -- DSR input
        DCDN        : in std_logic;                             -- DCD input
        RIN         : in std_logic;                             -- RI input
        SIN         : in std_logic;                             -- Receiver input
        SOUT        : out std_logic                             -- Transmitter output
    );
    end component;

    component slib_input_filter is
    generic (
        SIZE        : natural := 4      -- Filter counter size
    );
    port (
        CLK         : in std_logic;     -- Clock
        RST         : in std_logic;     -- Reset
        CE          : in std_logic;     -- Clock enable
        D           : in std_logic;     -- Signal input
        Q           : out std_logic     -- Signal output
    );
    end component;

    constant fragm  :  integer :=  ((1843200*1024)+(FPGACLOCK/128))/(FPGACLOCK/64);
    signal counter  :  unsigned (16 downto 0);

--    signal uartclk  : std_logic;
--    signal luartclk : std_logic;
    signal intcsel  : std_logic;
    signal sel_u1   : std_logic;
    signal sel_u2   : std_logic;
    signal cs_u1    : std_logic;
    signal cs_u2    : std_logic;
    signal cs_keyb  : std_logic;
    signal cs_par   : std_logic;
    signal cs_lcd   : std_logic;
    signal dout1    : std_logic_vector(7 downto 0);
    signal dout2    : std_logic_vector(7 downto 0);
    signal rclk1    : std_logic;
    signal rclk2    : std_logic;
    signal bdClk    : std_logic;  -- baud clock enable
    signal rst      : std_logic;
    signal u1r, u1w : std_logic;
    signal u2r, u2w : std_logic;
    signal u1addr   : std_logic_vector(2 downto 0);
    signal u1datain : std_logic_vector(7 downto 0);
    signal kbsreg   : std_logic_vector(10 downto 0);  -- IC17,18 (74HC164)
    signal kdatout  : std_logic_vector(7 downto 0); -- IC13 (74HC540)
    signal kctlout  : std_logic_vector(7 downto 0); -- IC16 (74HC541)
    signal kcontrol : std_logic_vector(4 downto 0); -- IC21 (74HC574)
    signal kclk,kdat: std_logic;
    signal sci,slci : std_logic;
    signal lkclk    : std_logic;
    signal ic15a    : std_logic;  -- 1/2 74HC74
    signal ic15b    : std_logic;  -- 1/2 74HC74
    signal ic14     : unsigned(7 downto 0);  -- 74HC393
    signal en4mhz   : std_logic;
    signal lclk4mhz : std_logic;
    signal ic23a    : std_logic;  -- 1/2 74HC74
    signal unsic22b : unsigned(2 downto 0);
    signal ic22b    : std_logic_vector(2 downto 0);  -- 1/2 74HC393
    signal nraw     : std_logic;  -- not (n_rd and n_wr)
    signal lnraw    : std_logic;  -- last nraw
    signal e1,e2    : std_logic;
    signal ed1,ed2  : std_logic;
    signal lcddout  : std_logic_vector(7 downto 0);
    signal dlyread  : std_logic;
    signal txd1,txd2: std_logic;
    signal ic34     : std_logic_vector(7 downto 0);
    signal n_rdlcd  : std_logic;
    signal n_wrlcd  : std_logic;
    signal syncdati : std_logic_vector(7 downto 0);
    signal n_swrite : std_logic;
    signal n_swrite2: std_logic;
    signal n_sioen1 : std_logic;
    signal wrpulse  : std_logic;
    signal uaddr    : std_logic_vector(10 downto 0) := (others => '0');

begin

    -- fetch input data and signals
    fetchindata:
    process (clk_i, n_reset_i)
    begin
        if n_reset_i = '0' then
            syncdati <= (others => '0');
            uaddr    <= (others => '0');
            n_swrite <= '1';
            n_swrite2<= '1';
            n_sioen1 <= '1';
        elsif rising_edge(clk_i) then
            syncdati <= data_i;
            uaddr    <= address_i(10 downto 0);
            n_swrite <= n_write_i;
            n_swrite2<= n_swrite;
            n_sioen1 <= n_ioen1_i;
        end if;
    end process;
    wrpulse <= (not n_swrite) and n_swrite2;

    -- chip select's
    cs_u1  <= '1' when (n_sioen1  = '0') and (uaddr(10 downto 7) = "0110") and (uaddr(3) = '0') else '0';
    cs_u2  <= '1' when (n_sioen1  = '0') and (uaddr(10 downto 7) = "0110") and (uaddr(3) = '1') else '0';
    cs_keyb<= '1' when (n_ioen1_i = '0') and (address_i(10 downto 7) = "0100") and (kb_ena_i = '1') else '0';
    cs_lcd <= '1' when (n_ioen1_i = '0') and (address_i(10 downto 7) = "0101") else '0';
    cs_par <= '1' when (n_ioen1_i = '0') and (address_i(10 downto 7) = "0111") and (address_i(1) = '1') else '0';
    
    intcsel <= cs_u1 or cs_u2 or cs_keyb or cs_lcd or cs_par;
    selected_o <= intcsel;

    -- uart signals
    rst <= not n_reset_i;
    u1r <= not n_read_i  when u1_extc_i = '0' else u1_read_i;
    u1w <= not n_swrite  when u1_extc_i = '0' else u1_write_i;
    u2r <= not n_read_i;
    u2w <= not n_swrite;
    u1addr   <= uaddr(2 downto 0) when u1_extc_i = '0' else u1_addr_i;
    u1datain <= syncdati          when u1_extc_i = '0' else u1_data_i;
    u1_data_o<= dout1;
    sel_u1   <= cs_u1 when u1_extc_i = '0' else '1';
    sel_u2   <= cs_u2 when u1_extc_i = '0' else cs_u1;

    -- include UART1    
    uart_1:
    uart_16750 port map (
        CLK     => clk_i,
        RST     => rst,
        BAUDCE  => bdClk,
        CS      => sel_u1,
        WR      => u1w,
        RD      => u1r,
        A       => u1addr,
        DIN     => u1datain,
        DOUT    => dout1,
        INT     => irqU1_o,
        OUT1N   => n_out1_o(0),
        OUT2N   => n_out2_o(0),
        RCLK    => rclk1,
        BAUDOUTN=> rclk1,
        RTSN    => n_rts_o(0),
        DTRN    => n_dtr_o(0),
        CTSN    => n_cts_i(0),
        DSRN    => n_dsr_i(0),
        DCDN    => n_dcd_i(0),
        RIN     => n_ri_i(0),
        SIN     => rxd_i(0),
        SOUT    => txd1
    );

    -- include UART2
    uart_2:
    uart_16750 port map (
        CLK     => clk_i,
        RST     => rst,
        BAUDCE  => bdClk,
        CS      => sel_u2,
        WR      => u2w,
        RD      => u2r,
        A       => uaddr(2 downto 0),
        DIN     => syncdati,
        DOUT    => dout2,
        INT     => irqU2_o,
        OUT1N   => n_out1_o(1),
        OUT2N   => n_out2_o(1),
        RCLK    => rclk2,
        BAUDOUTN=> rclk2,
        RTSN    => n_rts_o(1),
        DTRN    => n_dtr_o(1),
        CTSN    => n_cts_i(1),
        DSRN    => n_dsr_i(1),
        DCDN    => n_dcd_i(1),
        RIN     => n_ri_i(1),
        SIN     => rxd_i(1),
        SOUT    => txd2
    );

    -- assign output data
    dataout:
    process (intcsel, n_read_i, cs_u1, cs_u2, cs_keyb, address_i,
             kctlout, kdatout, dout1, dout2, lcddout, cs_lcd, cs_par, ic34, digi_i)
    begin
        dovalid_o <= not n_read_i;
        if (cs_keyb = '1') and (address_i(0) = '0') then
            data_o <= kctlout;
        elsif (cs_keyb = '1') and (address_i(0) = '1') then
            data_o <= kdatout;
        elsif cs_u1 = '1' then
            data_o <= dout1;
        elsif cs_u2 = '1' then
            data_o <= dout2;
        elsif cs_lcd = '1' then
            data_o <= lcddout;
        elsif (cs_par = '1') and (address_i(1 downto 0) = "10") then
            data_o <= ic34;
        elsif (cs_par = '1') and (address_i(1 downto 0) = "11") then
            data_o <= digi_i;
        else
            data_o <= (others => '1');
            dovalid_o <= '0';
        end if;
    end process;

    -- generate baud clock enable 1,843200 MHz
    bdproc:
    process (clk_i, n_reset_i)
    begin
        if n_reset_i = '0' then
            counter <= (others => '0');
            bdClk <= '0';
        elsif rising_edge(clk_i) then
            bdClk <= '0';
            if counter(16) = '0' then
                counter <= counter + fragm;
            else
                counter <= ('0' & counter(15 downto 0)) + fragm;
                bdClk <= '1';
            end if;
        end if;
    end process;

    -- -- generate baud clock enable 1,843200 MHz
    -- filtuclk: slib_input_filter generic map(3) port map(clk_i, rst, '1', uclk_i, uartclk);
    -- baudclken:
    -- process (clk_i, n_reset_i)
    -- begin
        -- if n_reset_i = '0' then
            -- luartclk <= '0';
        -- elsif rising_edge(clk_i) then
            -- luartclk <= uartclk;
        -- end if;
        -- bdClk <= uartclk and not luartclk;
    -- end process;
    
    -- synchronize txd with external baud clock
--    syntxd:
--    process (uclk_i, n_reset_i)
--    begin
--        if n_reset_i = '0' then
--            txd_o(0) <= '1';
--            txd_o(1) <= '1';
--        elsif rising_edge(uclk_i) then
            txd_o(0) <= txd1;
            txd_o(1) <= txd2;
--        end if;
--    end process;

    -- get 4MHz clock singal
    clk4edge:
    process (clk_i, n_reset_i, clk4MHz_i, lclk4mhz)
    begin
        if n_reset_i = '0' then
            lclk4mhz <= '0';
        elsif rising_edge(clk_i) then
            lclk4mhz <= clk4MHz_i;
        end if;
        en4mhz <= clk4MHz_i and not lclk4mhz;
    end process;


    -- filter PS2 inputs
    filt1: slib_input_filter generic map(7) port map(clk_i, rst, '1', kb_clk_i, kclk);
    filt2: slib_input_filter generic map(7) port map(clk_i, rst, '1', kb_dat_i, kdat);

    -- keyboard interface
    ps2ctrl:
    process (clk_i, n_reset_i, kbsreg, kctlout, kcontrol, kclk, kdat,
             ic15a, ic15b, sci, cs_keyb, address_i, n_write_i)
    begin
        -- IC13, 74HC540
        kdatout <= not kbsreg(8 downto 1);

        -- IC16, 74HC541
        kctlout(7) <= '0';
        kctlout(6) <= kcontrol(4);
        kctlout(5) <= kbsreg(9);
        kctlout(4) <= kbsreg(10);
        kctlout(3) <= kcontrol(3);
        kctlout(2) <= kcontrol(2);
        kctlout(1) <= kclk;
        kctlout(0) <= kdat;

        -- delay PS2 clock signal for edge detection
        if n_reset_i = '0' then
            lkclk <= '1';
        elsif rising_edge(clk_i) then
            lkclk <= kclk;
        end if;

        -- fetch write to IC21
        if n_reset_i = '0' then
            kcontrol <= (others => '0');
        elsif rising_edge(clk_i) then
            if (cs_keyb = '1') and (address_i(0) = '0') and (wrpulse = '1') then
                kcontrol(4) <= syncdati(7);
                kcontrol(3 downto 0) <= syncdati(3 downto 0);
            end if;
        end if;

        -- input shift register
        if (n_reset_i = '0') or (kcontrol(2) = '0') then
            kbsreg <= (others => '0');
            slci <= sci;
        elsif rising_edge(clk_i) then
            if (sci = '1') and (slci = '0') then
                kbsreg(9 downto 0) <= kbsreg(10 downto 1);
                kbsreg(10) <= not kdat;
            end if;
            slci <= sci;
        end if;
        sci <= not (kclk and (not kbsreg(0)));

        -- PS2 outputs
        if (kcontrol(1) = '0') or (ic15b = '1') then
            kb_clk_o <= '0';
        else
            kb_clk_o <= '1';
        end if;
        if kcontrol(0) = '0' then
            kb_dat_o <= '0';
        else
            kb_dat_o <= '1';
        end if;

        -- interrupt logic
        if (n_reset_i = '0') or (ic15b = '0') then
            ic14 <= (others => '0');
        elsif rising_edge(clk_i) then
            if en4mhz = '1' then
                ic14 <= ic14 + 1;
            end if;
        end if;
        if (n_reset_i = '0') or (ic15b = '0') then
            ic15a <= '0';
        elsif rising_edge(clk_i) then
            if ic14 = 255 then
                ic15a <= '1';
            end if;
        end if;
        if (n_reset_i = '0') or ((cs_keyb = '1') and
           (address_i(0) = '1') and (n_write_i = '0')) then
            ic15b <= '0';
        elsif rising_edge(clk_i) then
            if (lkclk = '0') and (kclk = '1') then
                ic15b <= kbsreg(0);
            end if;
        end if;
        irqKB_o <= ic15a and ic15b;

    end process;


    -- LCD interface
    lcdif:
    process (clk_i, n_reset_i, n_read_i, n_write_i, ic23a, ic22b, ic22b(2),
             unsic22b, cs_lcd, lcd_sw_i, e1, e2, ed1, ed2, data_i, address_i,
             dlyread, lcd_dat_i, n_rdlcd, n_wrlcd)
    begin
        -- IC22b
        if (n_reset_i = '0') or (ic23a = '1') then
            unsic22b <= (others => '0');
        elsif rising_edge(clk_i) then
            if en4mhz = '1' then
                unsic22b <= unsic22b + 1;
            end if;
        end if;
        ic22b <= std_logic_vector(unsic22b);

        -- IC23a
        nraw <= not (n_read_i and n_write_i);
        if n_reset_i = '0' then
            lnraw <= '1';
        elsif rising_edge(clk_i) then
            lnraw <= nraw;
        end if;
        if (n_reset_i = '0') then
            ic23a <= '1';
        elsif rising_edge(clk_i) then
            if ic22b = "110" then
                ic23a <= '1';
            else
                if (lnraw = '0') and (nraw = '1') then
                    ic23a <= not cs_lcd;
                end if;
            end if;
        end if;
        n_halt_o <= ic23a;

        -- delay n_read-signal and latch rs signal
        -- (this is additional in this FPGA version)
        if n_reset_i = '0' then
            dlyread <= '1';
            lcd_rs_o <= '0';
        elsif rising_edge(clk_i) then
            dlyread <= n_read_i;
            if cs_lcd = '1' then
                lcd_rs_o <= address_i(0);
            end if;
        end if;

        -- generate LCD read and write signals
        n_rdlcd <= n_read_i or dlyread;
        n_wrlcd <= not ((not ic23a) and ic22b(2));

        -- IC27
        e1 <= '0';
        e2 <= '0';
        if (cs_lcd = '1') then
            if (n_rdlcd = '0') or (n_wrlcd = '0') then
                if address_i(1) = '0' then
                    e1 <= '1';
                else
                    e2 <= '1';
                end if;
            end if;
        end if;

        -- LCD type select switches
        if (e2 = '1') and (lcd_sw_i /= "00") then
            lcddout <= (not lcd_sw_i) & lcd_dat_i(5 downto 0);
        else
            lcddout <= lcd_dat_i;
        end if;
  
        -- delay assertion of enable-outputs 
        -- (ensures the address-setup-time is met)
        if n_reset_i = '0' then
            ed1 <= '0';
            ed2 <= '0';
        elsif rising_edge(clk_i) then
            ed1 <= e1;
            ed2 <= e2;
        end if;
        lcd_e1_o <= e1 and ed1;
        lcd_e2_o <= e2 and ed2;
        
        -- other outputs
        lcd_dat_o  <= data_i;
        lcd_ddir_o <= not n_write_i;
        lcd_dren_o <= not cs_lcd;
        lcd_rw_o   <= n_write_i or not cs_lcd;

    end process;


    -- digital in/out
    process  (clk_i, n_reset_i, ic34)
    begin
        if n_reset_i = '0' then
            ic34 <= (others => '0');
        elsif rising_edge(clk_i) then
            if (cs_par = '1') and (address_i(1 downto 0) = "10") and (wrpulse = '1') then
                ic34 <= syncdati;
            end if;
        end if;
        digi_o <= ic34;
    end process;


end rtl;
