-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
--  Component:  Memory Unit
--

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;


entity memunit is
    port (
        -- Interface to CPU
        n_reset_i : IN  std_logic;
        clk_i     : IN  std_logic;
        address_i : IN  std_logic_vector(15 downto 0);
        selected_o: OUT std_logic;
        data_i    : IN  std_logic_vector(7 downto 0);
        data_o    : OUT std_logic_vector(7 downto 0);
        dovalid_o : OUT std_logic;
        n_ioen_o  : OUT std_logic_vector(2 downto 0);
        n_rdcod_i : IN  std_logic;
        n_read_i  : IN  std_logic;
        n_write_i : IN  std_logic;
        n_wren_i  : IN  std_logic;  -- internal /writeenable, can be tied to n_write_i
        cycstats_i: IN  std_logic_vector(15 downto 0);
        leds_o    : OUT std_logic_vector(3 downto 0);
        -- Interface to external memory chips
        memaddr_o : OUT std_logic_vector(20 downto 0);
        memdata_i : IN  std_logic_vector(7 downto 0);
        memdata_o : OUT std_logic_vector(7 downto 0);
        n_csram_o : OUT std_logic;
        n_csrom_o : OUT std_logic;
        n_mem_rd_o: OUT std_logic;
        n_mem_wr_o: OUT std_logic;
        -- Configuration Interface --
        doConfig_i: IN  std_logic;
        cfgExRam_i: IN  std_logic
    );
end memunit;


architecture rtl of memunit is

    signal romaddr  : std_logic_vector(17 downto 0);
    signal ramaddr  : std_logic_vector(20 downto 0);
    signal ram1adr  : std_logic_vector(14 downto 0);  -- address lines of 32kb RAM
    signal ram2adr  : std_logic_vector(18 downto 0);  -- address lines of 512kb RAM IC16
    signal ram3adr  : std_logic_vector(18 downto 0);  -- address lines of 512kb RAM IC29
    signal ioen     : std_logic_vector(2 downto 0);
    signal memio    : std_logic;  -- select memory bank registers
    signal csram1   : std_logic;  -- select 32kb RAM for lower 8kb
    signal csram2   : std_logic;  -- select 512kb RAM lo
    signal csram3   : std_logic;  -- select 512kb RAM hi
    signal csrom1   : std_logic;  -- select the 32kb EEPROM
    signal csrom2   : std_logic;  -- select the 128kb EEPROM
    signal oeram1   : std_logic;  -- output enable 32kb RAM
    signal oeram2   : std_logic;  -- output enable 512kb RAM lo
    signal oeram3   : std_logic;  -- output enable 512kb RAM hi
    signal csr3     : std_logic;  -- temp. signal for second 512kb RAM
    signal cszp,cssp: std_logic;  -- chip select zeropage/stackpage
    signal pa       : std_logic_vector(7 downto 0);  -- page number
    signal ra       : std_logic_vector(18 downto 14);-- ram3 addresses
    signal regrampg : std_logic_vector(7 downto 0);  -- REG_RAMPAGE    IC1, IC2
    signal regrompg : std_logic_vector(7 downto 0);  -- REG_ROMPAGE    IC3, IC4
    signal regstkpg : std_logic_vector(7 downto 0);  -- REG_STACKPAGE  IC22, IC23
    signal regzeropg: std_logic_vector(7 downto 0);  -- REG_ZEROPAGE   IC24, IC25
    signal rdrampg  : std_logic;
    signal rdrompg  : std_logic;
    signal rdzpage  : std_logic;
    signal rdspage  : std_logic;
    signal rdfreql  : std_logic;
    signal rdfreqh  : std_logic;
    signal wrrampg  : std_logic;
    signal wrrompg  : std_logic;
    signal wrzpage  : std_logic;
    signal wrspage  : std_logic;
    signal wrrdr    : std_logic;
    signal wrwsb    : std_logic;
    signal enrom    : std_logic;  -- FF, IC35A 74HC74
    signal wstbit   : std_logic;  -- FF, IC35B 74HC74
    signal xa14     : std_logic;
    signal enic30   : std_logic;
    signal enic31   : std_logic;
    signal enic32   : std_logic;
    signal t1,t2,t3 : std_logic;

begin

    -- decode I/O-banks (IC6 74AC138)
    ioen(0) <= '1' when address_i(15 downto 11) = "00100" else '0';
    ioen(1) <= '1' when address_i(15 downto 11) = "00101" else '0';
    ioen(2) <= '1' when address_i(15 downto 11) = "00110" else '0';
    memio   <= '1' when address_i(15 downto 11) = "00111" else '0';

    -- decode memory banks (IC9 74HC139)
    csram1  <= '1' when address_i(15 downto 13) = "000" else '0';
    csram2  <= '1' when (address_i(15 downto 14) /= "00") and (pa(5) = '0') else '0';
    csr3    <= '1' when (address_i(15 downto 14) /= "00") and (pa(5) = '1') else '0';
    
    -- decode zero/stack-page (IC26 74AC139)
    cszp    <= '1' when address_i(12 downto 8) = "00000" else '0';
    cssp    <= '1' when address_i(12 downto 8) = "00001" else '0';
    
    -- decode internal registers (IC7, IC8 74HC138)
    rdrampg <= '1' when (memio = '1') and (address_i(10 downto 8) = "000") and (n_read_i = '0') else '0';
    rdrompg <= '1' when (memio = '1') and (address_i(10 downto 8) = "001") and (n_read_i = '0') else '0';
    rdzpage <= '1' when (memio = '1') and (address_i(10 downto 8) = "010") and (n_read_i = '0') else '0';
    rdspage <= '1' when (memio = '1') and (address_i(10 downto 8) = "011") and (n_read_i = '0') else '0';
    wrrampg <= '1' when (memio = '1') and (address_i(10 downto 8) = "000") and (n_wren_i = '0') else '0';
    wrrompg <= '1' when (memio = '1') and (address_i(10 downto 8) = "001") and (n_wren_i = '0') else '0';
    wrzpage <= '1' when (memio = '1') and (address_i(10 downto 8) = "010") and (n_wren_i = '0') else '0';
    wrspage <= '1' when (memio = '1') and (address_i(10 downto 8) = "011") and (n_wren_i = '0') else '0';
    wrrdr   <= '1' when (memio = '1') and (address_i(10 downto 8) = "100") and (n_wren_i = '0') else '0';
    wrwsb   <= '1' when (memio = '1') and (address_i(10 downto 8) = "101") and (n_wren_i = '0') else '0';
    -- decode additional signals (only in FPGA version of MyCPU)
    rdfreql <= '1' when (memio = '1') and (address_i(10 downto 6) = "11110") and (address_i(0) = '0') and (n_read_i = '0') else '0';
    rdfreqh <= '1' when (memio = '1') and (address_i(10 downto 6) = "11110") and (address_i(0) = '1') and (n_read_i = '0') else '0';
    
    -- assign output signals
    n_ioen_o <= not ioen;
    leds_o   <= regrampg(3 downto 0) when address_i(15) = '0' else "0000";

    -- write to page registers
    pagereg:
    process (clk_i, n_reset_i)
    begin
        if n_reset_i = '0' then
            regrampg  <= (others => '0');
            regrompg  <= (others => '0');
            regstkpg  <= (others => '0');
            regzeropg <= (others => '0');
        elsif rising_edge(clk_i) then
            if wrrampg = '1' then
--report "RAMPAGE (POST-Code) now: " & integer'image(to_integer(unsigned(data_i)));  -- print post-code
                regrampg <= data_i;
            end if;
            if wrrompg = '1' then
                regrompg <= data_i;
            end if;
            if wrzpage = '1' then
                regzeropg <= enrom & data_i(6 downto 0);
            end if;
            if wrspage = '1' then
                regstkpg <= wstbit & data_i(6 downto 0);
            end if;
        end if;
    end process;

    -- write to en-rom and warmstart-bit
    wbits:  -- IC35 74HC74
    process (clk_i, n_reset_i)
    begin
        if n_reset_i = '0' then
            enrom  <= '1';
            wstbit <= '0';
        elsif rising_edge(clk_i) then
            if doConfig_i = '1' then
                enrom <= not cfgExRam_i;
            elsif wrrdr = '1' then
                enrom <= not (data_i(0) and data_i(1));
            end if;
            if wrwsb = '1' then
                wstbit <= data_i(0);
            end if;
        end if;
    end process;
    
    -- data read output
    process (rdrampg, rdrompg, rdzpage, rdspage,
             regrampg, regrompg, regzeropg, regstkpg,
             n_rdcod_i, csrom1, csrom2, memdata_i,
             rdfreql, rdfreqh, cycstats_i,
             csram1, csram2, csram3, 
             oeram1, oeram2, oeram3)
    begin
        dovalid_o <= '1';
        n_mem_rd_o <= '1';
        if rdrampg = '1' then
            data_o <= regrampg;
        elsif rdrompg = '1' then
            data_o <= regrompg;
        elsif rdzpage = '1' then
            data_o <= regzeropg;
        elsif rdspage = '1' then
            data_o <= regstkpg;
        elsif rdfreql = '1' then
            data_o <= cycstats_i(7 downto 0);
        elsif rdfreqh = '1' then
            data_o <= cycstats_i(15 downto 8);
        elsif (csrom1 = '1') and (n_rdcod_i = '0') then
            n_mem_rd_o <= '0';
            data_o <= memdata_i;
        elsif (csrom2 = '1') and (n_rdcod_i = '0') then
            n_mem_rd_o <= '0';
            data_o <= memdata_i;
        elsif (csram1 = '1') and (oeram1 = '1') then
            n_mem_rd_o <= '0';
            data_o <= memdata_i;
        elsif (csram2 = '1') and (oeram2 = '1') then
            n_mem_rd_o <= '0';
            data_o <= memdata_i;
        elsif (csram3 = '1') and (oeram3 = '1') then
            n_mem_rd_o <= '0';
            data_o <= memdata_i;
        else
            dovalid_o <= '0';
            data_o <= (others => '0');
        end if;
    end process;

    -- assign page number
    pa <= regrompg when address_i(15) = '1' else regrampg;
    
    -- decode chip-selects for EPROMs
    csrom1 <= (not n_rdcod_i) and enrom and (not address_i(15));
    csrom2 <= (not n_rdcod_i) and enrom and address_i(15) and pa(0);
    
    -- ROMs and RAMs
    romaddr <= '0' & pa(7 downto 6) & address_i(14 downto 0) when csrom2 = '1'
               else "100" & address_i(14 downto 0);
   
    ramaddr <= "100000" & ram1adr when (csram1 = '1') and (n_rdcod_i = '1') else
                   "01" & ram3adr when csram3 = '1' else
                   "00" & ram2adr when csram2 = '1' else
                (others => '0');  --error
    
    xa14 <= '1' when (address_i(15 downto 14) = "11") or ((address_i(15) = '0') and (pa(0) = '1')) else '0';
    
    ram1adr <= regzeropg(6 downto 0) & address_i(7 downto 0) when cszp = '1' else
               regstkpg(6 downto 0)  & address_i(7 downto 0) when cssp = '1' else
               "00" & address_i(12 downto 0);
    ram2adr <= pa(4 downto 1) & xa14 & address_i(13 downto 0);
    ram3adr <= ra & address_i(13 downto 0);

    -- glue logic from memory extension board    
    t1 <= enrom or address_i(15);
    t2 <= enrom or not (address_i(15) and pa(0));
    enic30 <= n_rdcod_i or (t1 and t2);
    enic31 <= not (t1 or n_rdcod_i);
    enic32 <= not (n_rdcod_i or t2);

    -- glue logic from memory base board
    t3 <= not (n_read_i and (n_rdcod_i or (not address_i(15)) or pa(0)));
    oeram2 <= t3;
    oeram1 <= not n_read_i;
    
    -- code for IC30, IC31, IC32    
    ra <= pa(4 downto 1) & xa14 when enic30 = '1' else
          "1011" & address_i(14) when enic31 = '1' else
          "11" & pa(7 downto 6) & address_i(14) when enic32 = '1' else
          "11111";
    csram3 <= csr3 when enic30 = '1' else '1';
    oeram3 <= t3   when enic30 = '1' else '1';

    -- assign output signals
    selected_o <= memio or csram1 or csram2 or csram3 or ((csrom1 or csrom2) and not n_rdcod_i);
    memaddr_o  <= "000" & romaddr when (csrom1 = '1') or (csrom2 = '1') else ramaddr;
    memdata_o  <= data_i;
    n_csram_o  <= not (csram1 or csram2 or csram3);
    n_csrom_o  <= not (csrom1 or csrom2);
    n_mem_wr_o <= n_write_i or not (csram1 or csram2 or csram3);
    
end rtl;

