-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
--  Component:  Interrupt Controller
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;


entity intctrl is
    port (
        n_reset_i : IN  std_logic;
        clk_i     : IN  std_logic;
        address_i : IN  std_logic_vector(15 downto 0);
        selected_o: OUT std_logic;
        data_i    : IN  std_logic_vector(7 downto 0);
        data_o    : OUT std_logic_vector(7 downto 0);
        dovalid_o : OUT std_logic;
        n_ioen1_i : IN  std_logic;
        n_read_i  : IN  std_logic;
        n_write_i : IN  std_logic;
        clk4MHz_i : IN  std_logic;
        irq_i     : IN  std_logic_vector(7 downto 0);
        irq_o     : OUT std_logic;
        tmrirq_o  : OUT std_logic
    );
end intctrl;


architecture rtl of intctrl is

    signal ctr16    : unsigned(15 downto 0); -- IC7, IC8
    signal intcsel  : std_logic;
    signal rd0, rd1 : std_logic;
    signal wr0, wr1 : std_logic;
    signal ien      : std_logic_vector(7 downto 0); -- IC2 74HC574
    signal ints     : std_logic_vector(7 downto 0); -- IC9-IC12
    signal sint     : std_logic_vector(7 downto 0);
    signal irqin    : std_logic_vector(7 downto 0);
    signal irqlast  : std_logic_vector(7 downto 0);
    signal last4mhz : std_logic; -- FF

begin

    -- IC5 (74HC138)
    intcsel <= '1' when (n_ioen1_i = '0') and (address_i(10 downto 7) = "0010") else '0';
    selected_o <= intcsel;

    -- IC6 (74HC139)
    rd0 <= '1' when (n_read_i  = '0') and (intcsel = '1') and (address_i(0) = '0') else '0';
    rd1 <= '1' when (n_read_i  = '0') and (intcsel = '1') and (address_i(0) = '1') else '0';
    wr0 <= '1' when (n_write_i = '0') and (intcsel = '1') and (address_i(0) = '0') else '0';
    wr1 <= '1' when (n_write_i = '0') and (intcsel = '1') and (address_i(0) = '1') else '0';

    ienreg:  -- IC2,IC3 (74HC574)
    process (clk_i, n_reset_i)
    begin
        if n_reset_i = '0' then
            ien <= (others => '0');
        elsif rising_edge(clk_i) then
            if wr1 = '1' then
                ien <= data_i;
            end if;
        end if;
    end process;

    -- assign data output (IC2, IC4)
    -- (changed to latched output on 2010-08-21)
    -- old code:
    --     data_o <= ints when rd0 = '1' else ien;
    --     dovalid_o <= rd0 or rd1;
    process (clk_i, n_reset_i)
    begin
        if n_reset_i = '0' then
            data_o <= (others => '0');
            dovalid_o <= '0';
        elsif rising_edge(clk_i) then
            if rd0 = '1' then
                data_o <= ints;
            else
                data_o <= ien;
            end if;
            dovalid_o <= rd0 or rd1;
        end if;
    end process;
    
    -- assign interrupt output (IC15, IC16)
    irq_o <= '1' when ints /= "00000000" else '0';

    -- mask interrupt inputs
    sint(0) <= irq_i(0);
    sint(7 downto 1) <= irq_i(7 downto 1) when wr0 = '0' else (others => '0');
    
    -- latch interrupts (IC9-IC12)
    intlatch:
    process (clk_i, n_reset_i)
    begin
        if n_reset_i = '0' then
            irqin   <= (others => '0');
            irqlast <= (others => '0');
            ints    <= (others => '0');
        elsif rising_edge(clk_i) then
            irqlast <= irqin;
            irqin   <= sint;
            for I in 0 to 7 loop
                if ien(I) = '0' then
                    ints(I) <= '0';
                elsif (irqlast(I) = '0') and (irqin(I) = '1') then
                    ints(I) <= '1';
                end if;
            end loop;
        end if;
    end process;

    -- generate the timer interrupt
    timerint:
    process (clk_i, clk4MHz_i, n_reset_i, ctr16)
        variable slv : std_logic_vector(15 downto 0);
    begin
        if n_reset_i = '0' then
            last4mhz <= '0';
            ctr16 <= (others => '0');
        elsif rising_edge(clk_i) then
            if (last4mhz = '0') and (clk4MHz_i = '1') then
                ctr16 <= ctr16 + 1;
            end if;
            last4mhz <= clk4MHz_i;
        end if;
        slv := std_logic_vector(ctr16);
        tmrirq_o <= slv(15);
    end process;

end rtl;
