-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
-- IDE interface: control logic
--
-- The main part of the IDE interface is placed in a separate CPLD.
-- This file implements the timing stuff for IDE poll mode.
-- Unfortunately the code does not fit into the CPLD so it is put into the FPGA.
--
-- The CPLD is connected to the FPGA via the external CPU bus. The CPLD has
-- one dedicated line to the FPGA: n_iderdy_i. The FPGA uses two of the
-- global address lines to control the CPLD:  A4 is used to mask the
-- IDE IOR/IOW signals, and A5 is used to switch between data bits
-- 0-7 and 8-15 on the data bus.
--
-- Dennis Kuschel, February 2010
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;


entity idecontrol is
    generic (
        FPGACLOCK : natural := 32000000   -- MHz
    );
    port (
        n_reset_i : IN  std_logic;  -- global reset
        clk_i     : IN  std_logic;  -- input clock (FPGA main clock)
        select_i  : IN  std_logic;  -- CPLD select line (shared with debug LED)
        n_read    : IN  std_logic;  -- data read signal from CPU core
        n_write   : IN  std_logic;  -- data write signal from CPU core
        address_i : IN  std_logic_vector(15 downto 0); -- from CPU core
        address_o : OUT std_logic_vector(15 downto 0); -- to CPLD
        data_i    : IN  std_logic_vector(7 downto 0);  -- from CPLD
        data_o    : OUT std_logic_vector(7 downto 0);  -- to CPU core
        n_iderdy_i: IN  std_logic;  -- direct line from CPLD
        halt_o    : OUT std_logic   -- to CPLD, inserts wait states
    );
end idecontrol;


architecture rtl of idecontrol is

    constant CLOCKS_SETUP : integer := (FPGACLOCK / 14285714) + 1;  -- ca.  70 ns
    constant CLOCKS_HOLD  : integer := (FPGACLOCK /  2777777) + 1;  -- ca. 290+70 ns
    constant CLOCKS_RDLO  : integer := (FPGACLOCK /  2083333) + 1;  -- ca. 120+290+70 ns
    constant CLOCKS_MAX   : integer := CLOCKS_RDLO + 1;
    signal   counter      : integer range 0 to CLOCKS_MAX;
    signal   upper_bits   : std_logic_vector(7 downto 0);
    signal   datalatch    : std_logic_vector(7 downto 0);
    signal   readupper    : std_logic;
    signal   n_diorw      : std_logic;
    signal   iderdy       : std_logic;
    signal   drivesel     : std_logic;
    signal   waitforend   : std_logic;

begin

    -- drivesel is asserted when CPU must do a slow IDE bus transfer
    drivesel <= '1' when (select_i = '1') and (address_i(8) = '0') else '0';

    -- read upper-bits register
    data_o <= upper_bits when (address_i(8) = '1') and (address_i(0) = '0') else
              datalatch  when drivesel = '1' else data_i;

    -- assign output address, add control signals
    address_o(15 downto 6) <= address_i(15 downto 6);
    address_o( 3 downto 0) <= address_i( 3 downto 0);
    address_o(4) <= n_diorw;
    address_o(5) <= readupper;

    -- synchronize n_iderdy_i signal
    sncrdy:
    process (n_reset_i, clk_i)
    begin
        if n_reset_i = '0' then
            iderdy <= '1';
        elsif rising_edge(clk_i) then
            iderdy <= (not n_iderdy_i) or (not select_i);
        end if;
    end process;

    -- bus time control
    btmctrl:
    process (n_reset_i, clk_i)
    begin
        if n_reset_i = '0' then
            counter    <= 0;
            n_diorw    <= '1';
            readupper  <= '1';
            halt_o     <= '0';
            waitforend <= '0';
            upper_bits <= (others => '0');
            datalatch  <= (others => '0');
        elsif rising_edge(clk_i) then
            if (drivesel = '0') or ((n_read = '1') and (n_write = '1')) then
                counter    <= 0;
                n_diorw    <= '1';
                readupper  <= '1';
                halt_o     <= '0';
                waitforend <= '0';
            else
                if waitforend = '0' then
                    halt_o  <= '1';
                    if (counter = CLOCKS_SETUP) then
                        n_diorw <= '0';
                    end if;
                    if ((counter = CLOCKS_HOLD) and (n_read = '0')) then
                        readupper  <= '0';
                        upper_bits <= data_i;
                    end if;
                    if ((counter = CLOCKS_HOLD) and (n_write = '0')) or
                       ((counter = CLOCKS_RDLO) and (n_read  = '0')) then
                       datalatch  <= data_i;
                       n_diorw    <= '1';
                       readupper  <= '1';
                       halt_o     <= '0';
                       waitforend <= '1';
                    else
                        if (counter /= CLOCKS_HOLD-1) or (iderdy = '1') then
                            counter <= counter + 1;
                        end if;
                    end if;
                end if;
            end if;
        end if;
    end process;

end rtl;
