-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
-- Flash ROM programming over RS232
--
-- Serial command format:
--    send 3 bytes:  cmd+adr / adr-hi / adr-lo
--    format:
--       byte1, bit7:   1=reset
--       byte1, bit6:   0=memory access, 1=command
--    when memory access:
--       byte1, bit5:   0=read,1=write
--       byte1, bit4:   flag: 0 = select SRAM, 1 = select Flash Memory
--       byte1, bit3:   address bit 19 (for RAM access only, ignored by Flash Memory)
--       byte1, bit2:   address bit 18
--       byte1, bit1:   address bit 17
--       byte1, bit0:   address bit 16
--       byte2      :   address bits 15..8
--       byte3      :   address bits  7..0
--    if write:
--       byte4      :   data byte
--    if read: one data byte is transmitted back
--
--    when command:
--       byte1, bit7:   0
--       byte1, bit6:   1
--       byte1, bit5-4: number of parameter N bytes following
--       byte1, bit3-0: command number
--       N more bytes (depending on command and bits 5-4 in byte1)
--
--    valid commands:
--       Byte1, Byte2
--        0x50,  --     NOP
--        0x51, 0x00    run MyCPU, execute code from RAM, com1:=not available(used for debug), com2:=com1(terminal)
--        0x51, 0x01    run MyCPU, execute code from ROM, com1:=not available(used for debug), com2:=com1(terminal)
--        0x51, 0x02    run MyCPU, execute code from RAM, disconnect debug interface
--        0x51, 0x03    run MyCPU, execute code from ROM, disconnect debug interface
--        0x51, 0x04    continue running MyCPU in last mode set
--        0x42,  --     stop running MyCPU
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;


entity flashprog is
    port (
        res_i     : IN  std_logic;
        clk_i     : IN  std_logic;
        enable_i  : IN  std_logic;
        --- UART1 external control ---
        urequest_o: OUT std_logic;     -- if '1' the flashprog entity requests exclusive access to the UART
        uaddr_o   : OUT std_logic_vector(2 downto 0);
        udata_o   : OUT std_logic_vector(7 downto 0);
        udata_i   : IN  std_logic_vector(7 downto 0);
        uread_o   : OUT std_logic;
        uwrite_o  : OUT std_logic;
        --- Flash Memory Interface ---
        mrequest_o: OUT std_logic_vector(1 downto 0);  -- if not "00" the flashprog entity requests exclusive access to the memory
        maddr_o   : OUT std_logic_vector(18 downto 0);
        mdata_o   : OUT std_logic_vector(7 downto 0);
        mdata_i   : IN  std_logic_vector(7 downto 0);
        mdvalid_o : OUT std_logic;
        mread_o   : OUT std_logic;
        mwrite_o  : OUT std_logic;
        --- CPU Interface ---
        cpuHalt_o : OUT std_logic;
        cpuReset_o: OUT std_logic;
        cpuFetch_i: IN  std_logic;
        -- Configuration Interface --
        doConfig_o: OUT std_logic;
        memCfg_o  : OUT std_logic   -- 0=execute code from ROM, 1=execute code from RAM
    );
end flashprog;


architecture rtl of flashprog is

    type state_type1 is (C_RESET, C_IDLE, C_START, C_CFG1, C_CFG2, C_CFG3, C_CFG4, C_CFG5, C_CFG6,
                         C_CFG7, C_CFG8, C_CFG9, C_IDLE1, C_IDLE2, C_IDLE3, C_IDLE4, C_IDLE5,
                         C_ADDR1, C_ADDR2, C_ADDR3, C_ADDR4, C_ADDR5, C_ADDR6, C_ADDR7,
                         C_WRITE1, C_WRITE2, C_WRITE3, C_WRITE4, C_WRITE5, C_WRITE6, C_WRITE7,
                         C_WRITE8, C_WRITE9, C_WRITEA, C_READ1, C_READ2, C_READ3, C_READ4, C_READ5,
                         C_PAR1A, C_PAR1B, C_PAR1C, C_PAR2A, C_PAR2B, C_PAR2C, C_PAR3A, C_PAR3B, C_PAR3C,
                         C_CMDEXEC, C_SENDACK, C_SENDNACK);
    type state_type2 is (U_IDLE, U_STEP1, U_STEP2, U_STEP3, U_STEP4, U_STEP5, U_STEP6);
    signal CState, NextCS : state_type1;
    signal UState, NextUS : state_type2;
    signal udata : std_logic_vector(7 downto 0);
    signal ubyte1: std_logic_vector(7 downto 0);
    signal param1: std_logic_vector(7 downto 0);
    signal param2: std_logic_vector(7 downto 0);
    signal param3: std_logic_vector(7 downto 0);
    signal reset : std_logic;
    signal do_read, do_write : std_logic;
    signal execCommand : std_logic;
    signal cmdACK      : std_logic;
    signal cmdNACK     : std_logic;
    signal mrequest    : std_logic_vector(1 downto 0);
    signal runCPU      : std_logic;
    signal stopCPU     : std_logic;
    signal CPUhalt     : std_logic;
    signal uartOff     : std_logic;
    signal doUartOff   : std_logic;
    
begin

    -- reset control
    doreset:
    process (clk_i, res_i)
    begin
        if res_i = '1' then
            reset <= '1';
        elsif rising_edge(clk_i) then
            reset <= not enable_i;
        end if;
    end process;
    urequest_o <= (not reset) and (not uartOff);

    -- memory chip select output
    mrequest   <= "00" when reset = '1' else      -- no memory chip selected
                  "01" when ubyte1(4) = '1' else  -- select flash memory
                  "11" when ubyte1(3) = '1' else  -- select SRAM No.2
                  "10";                           -- select SRAM No.1
    mrequest_o <= mrequest when (runCPU = '0') and (ubyte1(6) = '0') else "00";
    cpuHalt    <= '0' when reset  = '1' else
                  '0' when runCPU = '1' else
                  '1' when mrequest /= "00"
                  else '0';
    cpuHalt_o <= cpuHalt;

    -- read/write data from/to UART
    uartproc:
    process (clk_i, res_i, reset, do_read, do_write, UState)
    begin
        case UState is
            when U_IDLE   =>    if (do_read = '1') or (do_write = '1') then
                                    NextUS <= U_STEP1;
                                else
                                    NextUS <= U_IDLE;
                                end if;
            when U_STEP1  =>    NextUS <= U_STEP2;
            when U_STEP2  =>    NextUS <= U_STEP3;
            when U_STEP3  =>    if do_read = '0' then 
                                    NextUS <= U_IDLE;
                                else
                                    NextUS <= U_STEP4;
                                end if;
            when U_STEP4  =>    NextUS <= U_STEP5;
            when U_STEP5  =>    NextUS <= U_STEP6;
            when U_STEP6  =>    NextUS <= U_IDLE;
            when others   =>    NextUS <= U_IDLE;
        end case;

        if res_i = '1' then
            UState <= U_IDLE;
            udata <= (others => '0');
            uwrite_o <= '0';
            uread_o  <= '0';
            uartOff  <= '0';
        elsif rising_edge(clk_i) then
            if (reset = '1') or (uartOff = '1') then
                UState <= U_IDLE;
                uwrite_o <= '0';
                uread_o  <= '0';
            else
                uwrite_o <= '0';
                uread_o  <= '0';
                if (NextUS = U_STEP2) or (NextUS = U_STEP3) or
                   (NextUS = U_STEP4) or (NextUS = U_STEP5) then
                    if do_read = '1' then
                        uread_o <= '1';
                    elsif do_write = '1' then
                        if NextUS = U_STEP2 then
                            uwrite_o <= '1';
                        elsif NextUS = U_STEP3 then
                            uartOff <= doUartOff;
                        end if;
                    end if;
                end if;
                if (NextUS = U_STEP6) and (do_read = '1') then
                    udata <= udata_i;
                end if;
                UState <= NextUS;
            end if;
        end if;
    end process;


    -- main state machine
    flashprogctrl:
    process (clk_i, res_i, reset, CState, udata, ubyte1, mdata_i)
    begin

        do_read  <= '0';
        do_write <= '0';
        uaddr_o  <= "000";
        udata_o  <= (others => '0');

        case CState is
            when C_RESET    =>  NextCS <= C_CFG1;
            when C_CFG1     =>  -- write 0x80 to LCR
                                uaddr_o <= "011";
                                udata_o <= x"80";
                                do_write<= '1';
                                NextCS <= C_CFG2;
            when C_CFG2     =>  -- write 0x01 to DLL
                                uaddr_o <= "000";
                                udata_o <= x"01";
                                do_write<= '1';
                                NextCS <= C_CFG3;
            when C_CFG3     =>  -- write 0x00 to DLM
                                uaddr_o <= "001";
                                udata_o <= x"00";
                                do_write<= '1';
                                NextCS <= C_CFG4;
            when C_CFG4     =>  -- write 0x03 to LCR
                                uaddr_o <= "011";
                                udata_o <= x"03";
                                do_write<= '1';
                                NextCS <= C_CFG5;
            when C_CFG5     =>  -- write 0x03 to MCR
                                uaddr_o <= "100";
                                udata_o <= x"03";
                                do_write<= '1';
                                NextCS <= C_CFG6;
            when C_CFG6     =>  -- write 0x07 to FCR
                                uaddr_o <= "010";
                                udata_o <= x"07";
                                do_write<= '1';
                                NextCS <= C_CFG7;
            when C_CFG7     =>  -- read RBR
                                uaddr_o <= "000";
                                do_read<= '1';
                                NextCS <= C_CFG8;
            when C_CFG8     =>  -- read LSR
                                uaddr_o <= "101";
                                do_read<= '1';
                                NextCS <= C_CFG9;
            when C_CFG9     =>  if udata(3 downto 1) /= "000" then  -- check if error
                                    NextCS <= C_CFG7;
                                elsif udata(0) = '1' then  -- check if DR bit is set
                                    NextCS <= C_CFG7;
                                else
                                    NextCS <= C_IDLE1;
                                end if;
            when C_IDLE1    =>  -- read LSR
                                uaddr_o <= "101";
                                do_read<= '1';
                                NextCS <= C_IDLE2;
            when C_IDLE2    =>  if udata(3 downto 1) /= "000" then  -- check if error
                                    NextCS <= C_IDLE5;
                                elsif udata(0) = '1' then -- check if DR bit is set
                                    NextCS <= C_IDLE3;
                                else
                                    NextCS <= C_IDLE1;
                                end if;
            when C_IDLE3    =>  -- read RBR
                                uaddr_o <= "000";
                                do_read<= '1';
                                NextCS <= C_IDLE4;
            when C_IDLE4    =>  -- evaluate command
                                if udata(7) = '1' then
                                    NextCS <= C_IDLE1;  --reset
                                elsif udata(6) = '1' then
                                    -- branch to command execution if ubyte1 holds a command
                                    NextCS <= C_PAR1A;
                                else
                                    NextCS <= C_ADDR1;
                                end if;
            when C_IDLE5    =>  -- read RBR (clear error)
                                uaddr_o <= "000";
                                do_read<= '1';
                                NextCS <= C_IDLE1;
            when C_ADDR1    =>  -- read LSR
                                uaddr_o <= "101";
                                do_read<= '1';
                                NextCS <= C_ADDR2;
            when C_ADDR2    =>  if udata(3 downto 1) /= "000" then  -- check if error
                                    NextCS <= C_IDLE5;
                                elsif udata(0) = '1' then  -- check if DR bit is set
                                    NextCS <= C_ADDR3;
                                else
                                    NextCS <= C_ADDR1;
                                end if;
            when C_ADDR3    =>  -- read RBR
                                uaddr_o <= "000";
                                do_read<= '1';
                                NextCS <= C_ADDR4;
            when C_ADDR4    =>  -- read LSR
                                uaddr_o <= "101";
                                do_read<= '1';
                                NextCS <= C_ADDR5;
            when C_ADDR5    =>  if udata(3 downto 1) /= "000" then  -- check if error
                                    NextCS <= C_IDLE5;
                                elsif udata(0) = '1' then  -- check if DR bit is set
                                    NextCS <= C_ADDR6;
                                else
                                    NextCS <= C_ADDR4;
                                end if;
            when C_ADDR6    =>  -- read RBR
                                uaddr_o <= "000";
                                do_read<= '1';
                                NextCS <= C_ADDR7;
            when C_ADDR7    =>  -- check if read or write
                                if ubyte1(5) = '1' then
                                    NextCS <= C_WRITE1;
                                else
                                    NextCS <= C_READ1;
                                end if;
            when C_READ1    =>  NextCS <= C_READ2;
            when C_READ2    =>  NextCS <= C_READ3;
            when C_READ3    =>  NextCS <= C_READ4;
            when C_READ4    =>  -- send data byte to UART
                                uaddr_o <= "000";
                                udata_o <= mdata_i;
                                do_write <= '1';
                                NextCS <= C_READ5;
            when C_READ5    =>  NextCS <= C_IDLE1;
            when C_WRITE1   =>  -- read LSR
                                uaddr_o <= "101";
                                do_read<= '1';
                                NextCS <= C_WRITE2;
            when C_WRITE2   =>  if udata(3 downto 1) /= "000" then  -- check if error
                                    NextCS <= C_IDLE5;
                                elsif udata(0) = '1' then  -- check if DR bit is set
                                    NextCS <= C_WRITE3;
                                else
                                    NextCS <= C_WRITE1;
                                end if;
            when C_WRITE3   =>  -- read RBR
                                uaddr_o <= "000";
                                do_read<= '1';
                                NextCS <= C_WRITE4;
            when C_WRITE4   =>  -- write data to flash memory
                                NextCS <= C_WRITE5;
            when C_WRITE5   =>  NextCS <= C_WRITE6;
            when C_WRITE6   =>  NextCS <= C_WRITE7;
            when C_WRITE7   =>  NextCS <= C_WRITE8;
            when C_WRITE8   =>  NextCS <= C_WRITE9;
            when C_WRITE9   =>  NextCS <= C_WRITEA;
            when C_WRITEA   =>  NextCS <= C_IDLE1;
                                -- read parameters of command
            when C_PAR1A    =>  -- read LSR
                                if ubyte1(5 downto 4) = "00" then
                                    NextCS <= C_CMDEXEC;
                                else
                                    uaddr_o<= "101";
                                    do_read<= '1';
                                    NextCS <= C_PAR1B;
                                end if;
            when C_PAR1B    =>  if udata(3 downto 1) /= "000" then  -- check if error
                                    NextCS <= C_IDLE5;
                                elsif udata(0) = '1' then  -- check if DR bit is set
                                    NextCS <= C_PAR1C;
                                else
                                    NextCS <= C_PAR1A;
                                end if;
            when C_PAR1C    =>  -- read RBR
                                uaddr_o<= "000";
                                do_read<= '1';
                                NextCS <= C_PAR2A;
            when C_PAR2A    =>  -- read LSR
                                if ubyte1(5 downto 4) = "01" then
                                    NextCS <= C_CMDEXEC;
                                else
                                    uaddr_o<= "101";
                                    do_read<= '1';
                                    NextCS <= C_PAR2B;
                                end if;
            when C_PAR2B    =>  if udata(3 downto 1) /= "000" then  -- check if error
                                    NextCS <= C_IDLE5;
                                elsif udata(0) = '1' then  -- check if DR bit is set
                                    NextCS <= C_PAR2C;
                                else
                                    NextCS <= C_PAR2A;
                                end if;
            when C_PAR2C    =>  -- read RBR
                                uaddr_o <= "000";
                                do_read<= '1';
                                NextCS <= C_PAR3A;
            when C_PAR3A    =>  -- read LSR
                                if ubyte1(5 downto 4) = "10" then
                                    NextCS <= C_CMDEXEC;
                                else
                                    uaddr_o<= "101";
                                    do_read<= '1';
                                    NextCS <= C_PAR3B;
                                end if;
            when C_PAR3B    =>  if udata(3 downto 1) /= "000" then  -- check if error
                                    NextCS <= C_IDLE5;
                                elsif udata(0) = '1' then  -- check if DR bit is set
                                    NextCS <= C_PAR3C;
                                else
                                    NextCS <= C_PAR3A;
                                end if;
            when C_PAR3C    =>  -- read RBR
                                uaddr_o <= "000";
                                do_read<= '1';
                                NextCS <= C_CMDEXEC;

            when C_CMDEXEC  =>  -- wait until command is executed
                                if cmdACK = '1' then
                                    NextCS <= C_SENDACK;
                                elsif cmdNACK = '1' then
                                    NextCS <= C_SENDNACK;
                                else
                                    NextCS <= C_CMDEXEC;
                                end if;

            when C_SENDACK  =>  -- send an ACK (acknowledge)
                                uaddr_o <= "000";
                                udata_o <= x"A1";  -- ACK
                                do_write <= '1';
                                NextCS <= C_IDLE1;

            when C_SENDNACK =>  -- send a NACK (not acknowledge)
                                uaddr_o <= "000";
                                udata_o <= x"A0";  -- NACK
                                do_write <= '1';
                                NextCS <= C_IDLE1;

            when others     =>  NextCS <= C_RESET;
        end case;

        if res_i = '1' then
            CState <= C_RESET;
            ubyte1 <= (others => '0');
            maddr_o(15 downto 0) <= (others => '0');
            param1 <= (others => '0');
            param2 <= (others => '0');
            param3 <= (others => '0');
            mread_o  <= '0';
            mwrite_o <= '0';
            mdvalid_o<= '0';
            execCommand <= '0';
        elsif rising_edge(clk_i) then
            mread_o  <= '0';
            mwrite_o <= '0';
            mdvalid_o<= '0';
            execCommand <= '0';
            if reset = '1' then
                CState <= C_RESET;
            elsif NextUS = U_IDLE then
                CState <= NextCS;
            end if;
            if CState = C_IDLE4 then
                ubyte1 <= udata_i;
            end if;
            if CState = C_ADDR3 then
                maddr_o(15 downto 8) <= udata_i;
            end if;
            if CState = C_ADDR6 then
                maddr_o(7 downto 0) <= udata_i;
            end if;
            if (NextCS = C_READ1) or (NextCS = C_READ2) or
               (NextCS = C_READ3) or (NextCS = C_READ4) or (NextCS = C_READ5) then
                mread_o <= '1';
            end if;
            if (NextCS = C_WRITE5) or (NextCS = C_WRITE6) or
               (NextCS = C_WRITE7) or (NextCS = C_WRITE8) or (NextCS = C_WRITE9) then
                mwrite_o <= '1';
                mdvalid_o<= '1';
            elsif (NextCS = C_WRITE4) or (NextCS = C_WRITEA) then
                mdvalid_o<= '1';
            end if;
            if (NextCS = C_CMDEXEC) or (CState = C_CMDEXEC) then
                execCommand <= '1';
            end if;
            if CState = C_PAR1C then
                param1 <= udata_i;
            end if;
            if CState = C_PAR2C then
                param2 <= udata_i;
            end if;
            if CState = C_PAR3C then
                param3 <= udata_i;
            end if;
        end if;
        maddr_o(18 downto 16) <= ubyte1(2 downto 0);
        mdata_o <= udata;
    end process;

    -- execute commands
    cmdexec:
    process (clk_i, res_i)
    begin
        if res_i = '1' then
            cmdACK  <= '0';
            cmdNACK <= '0';
            cpuReset_o <= '0';
            doConfig_o <= '0';
            memCfg_o   <= '0';
            runCPU     <= '0';
            stopCPU    <= '0';
            doUartOff  <= '0';
        elsif rising_edge(clk_i) then
            cmdACK  <= '0';
            cmdNACK <= '0';
            cpuReset_o <= '0';
            doConfig_o <= '0';
            if (stopCPU = '1') and (cpuFetch_i = '1') then
                runCPU <= '0';
            end if;
            if runCPU = '0' then
                stopCPU <= '0';
            end if;
            if execCommand = '1' then
                if ubyte1(3 downto 0) = "0000" then  -- "NOP"
                    cmdACK <= '1';
                elsif ubyte1(5 downto 0) = "010001" then  -- run CPU, execute code from RAM or ROM
                    if param1 = x"00" then  -- execute code from ROM, swap serial interfaces, UART1 used for debug
                        cpuReset_o <= '1';
                        doConfig_o <= '1';
                        memCfg_o <= '0';
                        runCPU <= '1';
                        cmdACK <= '1';
                    elsif param1 = x"01" then  -- execute code from RAM, swap serial interfaces, UART1 used for debug
                        cpuReset_o <= '1';
                        doConfig_o <= '1';
                        memCfg_o <= '1';
                        runCPU <= '1';
                        cmdACK <= '1';
                    elsif param1 = x"02" then  -- execute code from ROM, disconnect debug interface
                        cpuReset_o <= '1';
                        doConfig_o <= '1';
                        memCfg_o <= '0';
                        runCPU <= '1';
                        cmdACK <= '1';
                        doUartOff <= '1';
                    elsif param1 = x"03" then  -- execute code from RAM, disconnect debug interface
                        cpuReset_o <= '1';
                        doConfig_o <= '1';
                        memCfg_o <= '1';
                        runCPU <= '1';
                        cmdACK <= '1';
                        doUartOff <= '1';
                    elsif param1 = x"04" then  -- continue to run
                        runCPU <= '1';
                        cmdACK <= '1';
                    else
                        cmdNACK <= '1';
                    end if;
                elsif ubyte1(5 downto 0) = "000010" then  -- stop CPU
                    if (cpuHalt = '0') and (runCPU = '1') then
                        stopCPU <= '1';
                    end if;
                    cmdACK <= '1';
                else
                    cmdNACK <= '1';
                end if;
            end if;
        end if;
    end process;

end rtl;
