-------------------------------------------------------------------------------
--
--  Copyright (c) 2010, Dennis Kuschel, www.mycpu.eu
--  All rights reserved. 
--
--  Redistribution and use in source and binary forms, with or without
--  modification, are permitted provided that the following conditions
--  are met:
--
--   1. Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--   2. Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in the
--      documentation and/or other materials provided with the distribution.
--   3. The name of the author may not be used to endorse or promote
--      products derived from this software without specific prior written
--      permission. 
--
--  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
--  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
--  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
--  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
--  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
--  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
--  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
--  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
--  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
--  OF THE POSSIBILITY OF SUCH DAMAGE.
--
-------------------------------------------------------------------------------
--
--  This file describes the functionality
--  of the CPLD on the MyCPUcompact board.
--  The CPLD contains the IDE interface logic
--  and a parallel printer port.
--
--  The time control portion of the IDE controller
--  is placed into the FPGA because the CPLD is
--  too small (it has only 72 macro cells).
--
--  Dennis Kuschel, February 2010
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;


entity cpld is
    port (
        -- processor bus signals
        N_RESET   : IN    std_logic;
        N_READ    : IN    std_logic;
        N_WRITE   : IN    std_logic;
        N_SELECT  : IN    std_logic;
        RDY       : INOUT std_logic;
        A         : IN    std_logic_vector(5 downto 0);
        A8        : IN    std_logic;
        A9        : IN    std_logic;
        D         : INOUT std_logic_vector(7 downto 0);
        -- IDE signals
        DD        : INOUT std_logic_vector(15 downto 0);
        DA        : OUT   std_logic_vector(2 downto 0);
        N_DRESET  : OUT   std_logic;
        N_DIOW    : OUT   std_logic;
        N_DIOR    : OUT   std_logic;
        IORDY     : IN    std_logic;
        N_IOCS16  : IN    std_logic;
        N_DASP0   : IN    std_logic;
        N_DASP1   : IN    std_logic;
        N_CS10    : OUT   std_logic;
        N_CS11    : OUT   std_logic;
        N_CS20    : OUT   std_logic;
        N_CS21    : OUT   std_logic;
        -- LPT signals
        PD        : OUT   std_logic_vector(7 downto 0);
        N_STROBE  : OUT   std_logic;
        N_AUTOFD  : OUT   std_logic;
        N_PRESET  : OUT   std_logic;
        N_SELOUT  : OUT   std_logic;
        BUSY      : IN    std_logic;
        PEMPTY    : IN    std_logic;
        SELIN     : IN    std_logic;
        N_ACK     : IN    std_logic;
        N_FAULT   : IN    std_logic;
        -- others
        SCL       : OUT   std_logic;
        SDA       : INOUT std_logic;
        LED       : OUT   std_logic
    );
end cpld;


architecture rtl of cpld is

    signal sel_ide      : std_logic;
    signal sel_lpt      : std_logic;
    signal dout_ide     : std_logic_vector(7 downto 0);
    signal dout_lpt     : std_logic_vector(7 downto 0);
    signal reg_lpt_data : std_logic_vector(7 downto 0);  -- register
    signal reg_lpt_ctrl : std_logic_vector(3 downto 0);  -- register
    signal reg_lpt_stat : std_logic_vector(7 downto 0);
    signal wr_lpt_data  : std_logic;
    signal wr_lpt_ctrl  : std_logic;
    signal bits16_out   : std_logic_vector(7 downto 0);  -- register (IC4)
    signal reg_ide_stat : std_logic_vector(7 downto 0);  -- line-driver (IC3)
    signal g_ide_drive  : std_logic;  -- enable: access register on ide drive
    signal g_ide_board  : std_logic;  -- enable: access register in CPLD
    signal wr_ide_ctrl  : std_logic;
    signal wr_ide_b16   : std_logic;
    signal f_iocs16     : std_logic;  -- rs-flip-flop
    signal isel, ires   : std_logic;
    signal or1, or2     : std_logic;
    signal idet         : std_logic;

begin

    ---------------------------------------------------------------------------
    --- connect LPT and IDE in CPLD                                         ---
    ---------------------------------------------------------------------------

    sel_ide <= '1' when (N_SELECT  = '0') and (A9 = '0') else '0';
    sel_lpt <= '1' when (N_SELECT  = '0') and (A9 = '1') and (A8 = '1') and (A(1) = '0') else '0';
    
    D <= (others => 'Z') when (N_READ = '1') or (N_RESET = '0') else
         dout_ide when sel_ide = '1' else
         dout_lpt when sel_lpt = '1' else
         (others => 'Z');


    ---------------------------------------------------------------------------
    --- Parallel Port (LPT)                                                 ---
    ---------------------------------------------------------------------------

    -- decode write-signals
    wr_lpt_data <= '0' when (sel_lpt = '1') and (A(0) = '0') and (N_WRITE = '0') else '1';
    wr_lpt_ctrl <= '0' when (sel_lpt = '1') and (A(0) = '1') and (N_WRITE = '0') else '1';

    -- write to registers
    proclpt:
    process (N_RESET, wr_lpt_data, wr_lpt_ctrl)
    begin
        if N_RESET = '0' then
            reg_lpt_data <= (others => '0');
        elsif rising_edge(wr_lpt_data) then
            reg_lpt_data <= D;
        end if;
        if N_RESET = '0' then
            reg_lpt_ctrl <= (others => '0');
        elsif rising_edge(wr_lpt_ctrl) then
            reg_lpt_ctrl <= D(3 downto 0);
        end if;
    end process;

    -- read from registers
    dout_lpt <= reg_lpt_data when A(1 downto 0) = "00" else reg_lpt_stat;

    -- assign to CPLD pins
    PD <= reg_lpt_data;
    N_STROBE <= reg_lpt_ctrl(0);
    N_AUTOFD <= reg_lpt_ctrl(1);
    N_PRESET <= reg_lpt_ctrl(2);
    N_SELOUT <= reg_lpt_ctrl(3);
    reg_lpt_stat(2 downto 0) <= reg_lpt_ctrl(2 downto 0);
    reg_lpt_stat(3) <= SELIN;
    reg_lpt_stat(4) <= N_FAULT;
    reg_lpt_stat(5) <= PEMPTY;
    reg_lpt_stat(6) <= N_ACK;
    reg_lpt_stat(7) <= BUSY;


    ---------------------------------------------------------------------------
    --- IDE Interface                                                       ---
    ---------------------------------------------------------------------------

    -- IC7B
    g_ide_drive <= '1' when (sel_ide = '1') and (A8 = '0') else '0';
    g_ide_board <= '1' when (sel_ide = '1') and (A8 = '1') else '0';

    -- IC8A
    wr_ide_b16  <= '0' when (g_ide_board = '1') and (N_WRITE = '0') and (A(0) = '0') else '1';
    wr_ide_ctrl <= '0' when (g_ide_board = '1') and (N_WRITE = '0') and (A(0) = '1') else '1';

    -- read from registers
    dout_ide <= reg_ide_stat    when g_ide_board = '1' else
                DD(15 downto 8) when A(5)        = '1' else
                DD(7 downto 0); -- when (g_ide_drive = '1')

    -- write to registers
    idereg:
    process (N_RESET, wr_ide_b16, wr_ide_ctrl)
    begin
        -- IC2
        if N_RESET = '0' then
            isel <= '0';
            ires <= '0';
            idet <= '0';
            or2  <= '0';
            or1  <= '0';
        elsif rising_edge(wr_ide_ctrl) then
            isel <= D(7);
            ires <= D(6);
            idet <= D(5);
            or2  <= D(2);
            or1  <= D(1);
        end if;
        -- IC4
        if N_RESET = '0' then
            bits16_out <= (others => '0');
        elsif rising_edge(wr_ide_b16) then
            bits16_out <= D;
        end if;
    end process;

    -- IC11B (74HC74, RS-Flip-Flop)
    f_iocs16 <= '1' when N_IOCS16 = '0' else
                '0' when (wr_ide_b16 = '0') or (N_RESET = '0') else f_iocs16;
    
    -- status register IC3
    reg_ide_stat(7) <= f_iocs16;
    reg_ide_stat(6) <= N_DASP1;
    reg_ide_stat(5) <= N_DASP0;
    reg_ide_stat(4) <= idet;
    reg_ide_stat(3) <= idet;
    reg_ide_stat(2) <= SDA;
    reg_ide_stat(1) <= '0';
    reg_ide_stat(0) <= '0';

    -- I2C-Bus
    SCL <= or2;
    SDA <= '0' when or1 = '0' else 'Z';

    -- data bus to IDE drive
    DD <= bits16_out & D when (g_ide_drive = '1') and (N_WRITE = '0') else
          (others => 'Z');

    -- misc. IDE output signals
    DA     <= A(2 downto 0);
    LED    <= (not N_DASP0) or (not N_DASP1);
    N_CS10 <= '0' when (g_ide_drive = '1') and (isel = '0') and (A(3) = '0') else '1';
    N_CS11 <= '0' when (g_ide_drive = '1') and (isel = '0') and (A(3) = '1') else '1';
    N_CS20 <= '0' when (g_ide_drive = '1') and (isel = '1') and (A(3) = '0') else '1';
    N_CS21 <= '0' when (g_ide_drive = '1') and (isel = '1') and (A(3) = '1') else '1';
    N_DIOW <=A(4) when (g_ide_drive = '1') and (N_RESET = '1') and (N_WRITE = '0') else '1';
    N_DIOR <=A(4) when (g_ide_drive = '1') and (N_RESET = '1') and (N_READ  = '0') else '1';
    N_DRESET <= N_RESET and ires;

    -- "IDE ready" -signal to FPGA
    RDY <= '0' when (N_RESET = '1') and (g_ide_drive = '1') and (IORDY = '1') else 'Z';

end rtl;


