/*-----------------------------------------------------------------------------
 *
 *   This tool converts the microcode EPROM images IC1.bin, IC2.bin and IC3.bin
 *   into synthesizeable vhdl code.
 *
 *   (c) in February 2010 by Dennis Kuschel, Germany
 *
 *   visit http://www.mycpu.eu for more information
 *
 *-----------------------------------------------------------------------------
 *
 *  Copyright (c) 2010, Dennis Kuschel.
 *  All rights reserved. 
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *   1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *   3. The name of the author may not be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission. 
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 *  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 *  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 *  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 *  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 *  OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *---------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "mcdefs.h"

#define EP_CLOCKS   (0xF100 | MC1_CLOCKS)             /* bit mask */
#define EP_OUT_EN   (0x4100 | MC1_OUTENABLES)         /* bit mask */
#define EP_ALU_MODE (0xF200 | MC2_ALU_MODE_SELECT)    /* ALU: mode select */
#define EP_ALU_FSEL (0x1200 | MC2_ALU_FL_REG_SELECT)  /* ALU: ALU-Flags(0) / Register-Flags(1) -Select */
#define EP_ALU_CSEL (0x1200 | MC2_ALU_CARRY_SELECT)   /* ALU: Carry select, 0 = visible, 1 = hidden */
#define EP_ALU_LDFC (0x1200 | MC2_ALU_LOAD_FLAG_C)    /* ALU: load carry flag */
#define EP_ALU_LDVZ (0x1200 | MC2_ALU_LOAD_FLAG_VZ)   /* ALU: load sign and zero flag */
#define EP_CONST_01 (0x1300 | MC3_CONST_01)           /* CONSTANT: 00 / 01 */
#define EP_CONST_02 (0x1300 | MC3_CONST_02)           /* CONSTANT: 00 / 02 */
#define EP_INTERN   (0x1300 | MC3_INTERN)             /* 1 = only internal process, no ext.bus access in next clock cycle */
#define EP_LOAD_PC  (0x1300 | MC3_LOAD_PC)            /* Load Program Counter (lo active) */
#define EP_ADRSEL   (0x1300 | MC3_ADR_SELECT)         /* 1 = temp.Addr, 0 = PC-Addr */
#define EP_ALUCLK   (0x1300 | MC3_ALUCLOCK)
#define EP_INCPC    (0x1300 | MC3_INC_PC)
#define EP_MEND     (0x1300 | MC3_MCODE_END)          /* Microcode end */

#define SO_CLOCKS   "clocks_o    "  /* std_logic_vector(3 downto 0) */
#define SO_OUT_EN   "enable_o    "  /* std_logic_vector(3 downto 0) */
#define SO_ALU_MODE "alu_mode_o  "  /* std_logic_vector(3 downto 0) */
#define SO_ALU_FSEL "alu_fsel_o  "  /* std_logic */
#define SO_ALU_CSEL "alu_csel_o  "  /* std_logic */
#define SO_ALU_LDFC "n_alu_ldfc_o"  /* std_logic */
#define SO_ALU_LDVZ "n_alu_ldvz_o"  /* std_logic */
#define SO_CONST_01 "const_01_o  "  /* std_logic */
#define SO_CONST_02 "const_02_o  "  /* std_logic */
#define SO_INTERN   "intern_o    "  /* std_logic */
#define SO_LOAD_PC  "n_loadpc_o  "  /* std_logic */
#define SO_ADRSEL   "adrsel_o    "  /* std_logic */
#define SO_ALUCLK   "n_aluclk_o  "  /* std_logic */
#define SO_INCPC    "n_incpc_o   "  /* std_logic */
#define SO_MEND     "n_mend_o    "  /* std_logic */

#define SI_OPCODE   "opcode_i"    /* std_logic_vector(7 downto 0) */
#define SI_IRQ      "irq_i"       /* std_logic */
#define SI_CFLAG    "cflag_i"     /* std_logic */
#define SI_ZFLAG    "zflag_i"     /* std_logic */
#define SI_VFLAG    "vflag_i"     /* std_logic */
#define SI_MCTR     "mctr_i"      /* std_logic_vector(4 downto 0) */


/* define default output levels */
unsigned char defo_clocks   = 0x0;
unsigned char defo_out_en   = 0x0;
unsigned char defo_alu_mode = 0x0;
unsigned char defo_alu_fsel = 0;
unsigned char defo_alu_csel = 0;
unsigned char defo_alu_ldfc = 1;
unsigned char defo_alu_ldvz = 1;
unsigned char defo_const_01 = 0;
unsigned char defo_const_02 = 0;
unsigned char defo_intern   = 0;
unsigned char defo_load_pc  = 1;
unsigned char defo_adrsel   = 1;
unsigned char defo_aluclk   = 1;
unsigned char defo_incpc    = 1;
unsigned char defo_mend     = 1;  /* negated logic */



#define EPROMSIZE (128L*1024)
unsigned char *eprom[3];

/* input signals to EPROMS */
unsigned opcode = 0;   // 0-255
unsigned irq = 0;
unsigned cflag = 0;
unsigned zflag = 0;
unsigned vflag = 0;
unsigned mctr = 0;     // 0-31

int tablevel_g = 0; /* number of tabulators for output */
int simu_code_g = 0; /* flag: if true, create code for vhdl simulation */
FILE *outf_g   = NULL; /* output file */


static inline unsigned epromadr(void)
{
  unsigned adr = (opcode << EPROM_OPCODE) +
                 (mctr << EPROM_COUNTER) +
                 (cflag << EPROM_FLAGS) +
                 (zflag << (EPROM_FLAGS+1)) +
                 (vflag << (EPROM_FLAGS+2)) +
                 (irq << EPROM_IRQLINE);
  return adr;
}


/* checks if flag has an effect for eprom outputs.
   flag can be cflag, zflag, vflag */
int isEqual(unsigned *flag)
{
  unsigned char e1, e2, e3;
  unsigned old, adr;
  int eq = 1;
  old = *flag;
  *flag = 0;
  adr = epromadr();
  e1 = eprom[0][adr];
  e2 = eprom[1][adr];
  e3 = eprom[2][adr];
  *flag = 1;
  adr = epromadr();
  if (e1 != eprom[0][adr])
    eq = 0;
  if (e2 != eprom[1][adr])
    eq = 0;
  if (e3 != eprom[2][adr])
    eq = 0;
  *flag = old;
  return eq;
}


unsigned char getSignal(unsigned sig)
{
  unsigned e = ((sig >> 8) & 3) - 1;
  unsigned char v;
  v = eprom[e][epromadr()] & ((unsigned char)(sig&0xFF));
  if (((sig & 0xf000) == 0x1000) && (v != 0))
    return 1;
  if ((sig & 0xf000) != 0xf000)
    return v >> (sig >> 12);
  return v;
}


/* output a line of text */
void emit(const char *str)
{
  int i = tablevel_g;
  if (*str)
    while (i-- > 0)  fprintf(outf_g, "\t");
  fprintf(outf_g, "%s\n", str);
}


char* genvector(char *buf, unsigned char val, int len)
{
  int i;
  if (len > 1) {
    buf[0] = 0x22;
  } else {
    buf[0] = 0x27;
    val = val ? 1 : 0;
  }
  for (i=0;i<len;i++) {
    buf[len-i] = (val & (1<<i)) ? '1' : '0';
  }
  buf[len+1] = (len > 1) ? 0x22 : 0x27;
  buf[len+2] = 0;
  return buf;
}


void outputDefaults(void)
{
  char buf[100];
  char v[20];

  sprintf(buf, SO_CLOCKS " <= %s;", genvector(v, defo_clocks, 4));
  emit(buf);
  sprintf(buf, SO_OUT_EN " <= %s;", genvector(v, defo_out_en, 4));
  emit(buf);
  sprintf(buf, SO_ALU_MODE " <= %s;", genvector(v, defo_alu_mode, 4));
  emit(buf);
  sprintf(buf, SO_ALU_FSEL " <= %s;", genvector(v, defo_alu_fsel, 1));
  emit(buf);
  sprintf(buf, SO_ALU_CSEL " <= %s;", genvector(v, defo_alu_csel, 1));
  emit(buf);
  sprintf(buf, SO_ALU_LDFC " <= %s;", genvector(v, defo_alu_ldfc, 1));
  emit(buf);
  sprintf(buf, SO_ALU_LDVZ " <= %s;", genvector(v, defo_alu_ldvz, 1));
  emit(buf);
  sprintf(buf, SO_CONST_01 " <= %s;", genvector(v, defo_const_01, 1));
  emit(buf);
  sprintf(buf, SO_CONST_02 " <= %s;", genvector(v, defo_const_02, 1));
  emit(buf);
  sprintf(buf, SO_INTERN " <= %s;", genvector(v, defo_intern, 1));
  emit(buf);
  sprintf(buf, SO_LOAD_PC " <= %s;", genvector(v, defo_load_pc, 1));
  emit(buf);
  sprintf(buf, SO_ADRSEL " <= %s;", genvector(v, defo_adrsel, 1));
  emit(buf);
  sprintf(buf, SO_ALUCLK " <= %s;", genvector(v, defo_aluclk, 1));
  emit(buf);
  sprintf(buf, SO_INCPC " <= %s;", genvector(v, defo_incpc, 1));
  emit(buf);
  sprintf(buf, SO_MEND " <= %s;", genvector(v, defo_mend, 1));
  emit(buf);
}


void outputSignals(void)
{
  char buf[100];
  char v[20];
  unsigned char s;

  if ((s = getSignal(EP_CLOCKS)) != defo_clocks) {
    sprintf(buf, SO_CLOCKS " <= %s;", genvector(v, s, 4));
    emit(buf);
  }
  if ((s = getSignal(EP_OUT_EN)) != defo_out_en) {
    sprintf(buf, SO_OUT_EN " <= %s;", genvector(v, s, 4));
    emit(buf);
  }
  if ((s = getSignal(EP_ALU_MODE)) != defo_alu_mode) {
    sprintf(buf, SO_ALU_MODE " <= %s;", genvector(v, s, 4));
    emit(buf);
  }
  if ((s = getSignal(EP_ALU_FSEL)) != defo_alu_fsel) {
    sprintf(buf, SO_ALU_FSEL " <= %s;", genvector(v, s, 1));
    emit(buf);
  }
  if ((s = getSignal(EP_ALU_CSEL)) != defo_alu_csel) {
    sprintf(buf, SO_ALU_CSEL " <= %s;", genvector(v, s, 1));
    emit(buf);
  }
  if ((s = getSignal(EP_ALU_LDFC)) != defo_alu_ldfc) {
    sprintf(buf, SO_ALU_LDFC " <= %s;", genvector(v, s, 1));
    emit(buf);
  }
  if ((s = getSignal(EP_ALU_LDVZ)) != defo_alu_ldvz) {
    sprintf(buf, SO_ALU_LDVZ " <= %s;", genvector(v, s, 1));
    emit(buf);
  }
  if ((s = getSignal(EP_CONST_01)) != defo_const_01) {
    sprintf(buf, SO_CONST_01 " <= %s;", genvector(v, s, 1));
    emit(buf);
  }
  if ((s = getSignal(EP_CONST_02)) != defo_const_02) {
    sprintf(buf, SO_CONST_02 " <= %s;", genvector(v, s, 1));
    emit(buf);
  }
  if ((s = getSignal(EP_INTERN)) != defo_intern) {
    sprintf(buf, SO_INTERN " <= %s;", genvector(v, s, 1));
    emit(buf);
  }
  if ((s = getSignal(EP_LOAD_PC)) != defo_load_pc) {
    sprintf(buf, SO_LOAD_PC " <= %s;", genvector(v, s, 1));
    emit(buf);
  }
  if ((s = getSignal(EP_ADRSEL)) != defo_adrsel) {
    sprintf(buf, SO_ADRSEL " <= %s;", genvector(v, s, 1));
    emit(buf);
  }
  if ((s = getSignal(EP_ALUCLK)) != defo_aluclk) {
    sprintf(buf, SO_ALUCLK " <= %s;", genvector(v, s, 1));
    emit(buf);
  }
  if ((s = getSignal(EP_INCPC)) != defo_incpc) {
    sprintf(buf, SO_INCPC " <= %s;", genvector(v, s, 1));
    emit(buf);
  }
  if ((s = getSignal(EP_MEND)) != defo_mend) {
    sprintf(buf, SO_MEND " <= %s;", genvector(v, s, 1));
    emit(buf);
  }
}


void outputOpCode(void)
{
  char buf[100];
  char v[20];
  unsigned endbits = 0;
  unsigned cntmax, mc, mask, th, e;

  emit("case " SI_MCTR " is");
  tablevel_g++;

  /* prepare for adding don't cares */
  for (mctr=0; mctr<30; mctr++)
  {
    cflag = 0;
    if (!getSignal(EP_MEND))
      endbits |= 1;
    cflag = 1;
    if (!getSignal(EP_MEND))
      endbits |= 2;
    zflag = 0;
    if (!getSignal(EP_MEND))
      endbits |= 4;
    zflag = 1;
    if (!getSignal(EP_MEND))
      endbits |= 8;
    vflag = 0;
    if (!getSignal(EP_MEND))
      endbits |= 16;
    vflag = 1;
    if (!getSignal(EP_MEND))
      endbits |= 32;
    if (endbits == 63)
      break;
  }
  cntmax = mctr+1;
  th = MCODES_PER_OP;
  while (th > cntmax)
    th >>= 1;

  if (cntmax <= 7)
    mask = 3;
  else
  if (cntmax <= 15)
    mask = 7;
  else
    mask = 15;

  e = MCODES_PER_OP;
  if ((cntmax == 8) || (cntmax == 12) || (cntmax == 16))
    e = cntmax;

  for (mc=0; mc<MCODES_PER_OP; mc++)
  {
    mctr = mc;
    if (mctr > cntmax)
    {
      if (simu_code_g)
        break;

      /* add some don't cares to the code to simplyfy the netlist */
      if (mctr >= e)
      {
        mctr = e;
      }
      else
      {
        mctr &= mask;
      
        while (mctr > cntmax)
        {
          mask >>= 1;
          mctr &= mask;
        }

        if (mctr >= th)
          break;
      }
    }
 
    if (mc > cntmax) {
      /* "don't cares" are written with capital letters */
      sprintf(buf, "WHEN %s =>", genvector(v, (unsigned char)mc, 5));
    } else {
      sprintf(buf, "when %s =>", genvector(v, (unsigned char)mc, 5));
    }
    emit(buf);
   
    tablevel_g++;
    if (!isEqual(&cflag))
    {
      if (!isEqual(&zflag) || !isEqual(&vflag)) {
        printf("\nmultiple flag changes not supported!\n");
        exit(1);
      }
      emit("if " SI_CFLAG " = '0' then");
      tablevel_g++;
      cflag = 0;
      outputSignals();
      tablevel_g--;
      emit("else");
      tablevel_g++;
      cflag = 1;
      outputSignals();
      tablevel_g--;
      emit("end if;");
    }
    else
    if (!isEqual(&zflag))
    {
      if (!isEqual(&cflag) || !isEqual(&vflag)) {
        printf("\nmultiple flag changes not supported!\n");
        exit(1);
      }
      emit("if " SI_ZFLAG " = '0' then");
      tablevel_g++;
      zflag = 0;
      outputSignals();
      tablevel_g--;
      emit("else");
      tablevel_g++;
      zflag = 1;
      outputSignals();
      tablevel_g--;
      emit("end if;");
    }
    else
    if (!isEqual(&vflag))
    {
      if (!isEqual(&zflag) || !isEqual(&cflag)) {
        printf("\nmultiple flag changes not supported!\n");
        exit(1);
      }
      emit("if " SI_VFLAG " = '0' then");
      tablevel_g++;
      vflag = 0;
      outputSignals();
      tablevel_g--;
      emit("else");
      tablevel_g++;
      vflag = 1;
      outputSignals();
      tablevel_g--;
      emit("end if;");
    }
    else
    {
      outputSignals();
    }
    tablevel_g--;
  }

  emit("WHEN others =>");
  tablevel_g--;
  emit("end case;");
}


void outputAllOPs(void)
{
  char buf[100];

  emit("case " SI_OPCODE " is");
  tablevel_g++;

  for (opcode=0; opcode<256; opcode++)
  {
    emit("");
    sprintf(buf, "when x\x22%02X\x22 =>", opcode);
    emit(buf);
    tablevel_g++;
    outputOpCode();
    tablevel_g--;
  }

  emit("when others =>");
  tablevel_g--;
  emit("end case;");
}


void outputRomCode(void)
{
  emit("--");
  emit("-- Emulation of MyCPU micro-code EPROMs IC1, IC2 and IC3");
  emit("--");
  emit("-- This file was automatically generated from the ICx.bin EPROM image files.");
  emit("--");
  if (!simu_code_g)
  {
    emit("-- WHEN's that are written in capital letters are \"don't cares\"");
    emit("--");
  }
  emit("");
  emit("library IEEE;");
  emit("use IEEE.std_logic_1164.all;");
  emit("use IEEE.std_logic_arith.all;");
  emit("");
  emit("entity ucode is");
  tablevel_g++;
  emit("port (");
  tablevel_g++;
  emit(SI_OPCODE"     : IN  std_logic_vector(7 downto 0);");
  emit(SI_MCTR"       : IN  std_logic_vector(4 downto 0);");
  emit(SI_IRQ"        : IN  std_logic;");
  emit(SI_CFLAG"      : IN  std_logic;");
  emit(SI_ZFLAG"      : IN  std_logic;");
  emit(SI_VFLAG"      : IN  std_logic;");
  emit(SO_CLOCKS    " : OUT std_logic_vector(3 downto 0);");
  emit(SO_OUT_EN    " : OUT std_logic_vector(3 downto 0);");
  emit(SO_ALU_MODE  " : OUT std_logic_vector(3 downto 0);");
  emit(SO_ALU_FSEL  " : OUT std_logic;");
  emit(SO_ALU_CSEL  " : OUT std_logic;");
  emit(SO_ALU_LDFC  " : OUT std_logic;");
  emit(SO_ALU_LDVZ  " : OUT std_logic;");
  emit(SO_CONST_01  " : OUT std_logic;");
  emit(SO_CONST_02  " : OUT std_logic;");
  emit(SO_INTERN    " : OUT std_logic;");
  emit(SO_LOAD_PC   " : OUT std_logic;");
  emit(SO_ADRSEL    " : OUT std_logic;");
  emit(SO_ALUCLK    " : OUT std_logic;");
  emit(SO_INCPC     " : OUT std_logic;");
  emit(SO_MEND      " : OUT std_logic");
  tablevel_g--;
  emit(");");
  tablevel_g--;
  emit("end ucode;");
  emit("");
  emit("architecture rtl of ucode is");
  emit("begin");
  emit("");
  tablevel_g++;
  emit("opdecode:");
  emit("process ("SI_OPCODE","SI_IRQ","SI_CFLAG","SI_ZFLAG","SI_VFLAG","SI_MCTR")");
  emit("begin");
  emit("");
  tablevel_g++;
  emit("-- default output values");
  outputDefaults();
  emit("");
  emit("if " SI_IRQ " = '1' then");
  tablevel_g++;
  emit("");
  emit("-- microcode for interrupt execution");
  irq = 1;
  outputOpCode();
  tablevel_g--;
  emit("else");
  tablevel_g++;
  emit("");
  emit("-- microcode for every OP-code");
  irq = 0;
  outputAllOPs();
  tablevel_g--;
  emit("end if;");
  tablevel_g--;
  emit("end process;");
  tablevel_g--;
  emit("");
  emit("end rtl;");
}


int loadimages(void)
{
  const char *fmt[3] = { "mcode%i.bin", "IC%i.bin", "IC%i_27c010.bin" };
  FILE *f;
  int i,r,l,sum,n=0;
  char name[64];

  for (i=0;i<3;i++)
  {
    fflush(stdout);
    eprom[i] = malloc(EPROMSIZE);
    if (!eprom[i]) {
      printf("loading EPROM image %i  ERROR (failed to allocate memory)\n", i+1);
      return -1;
    }
    do {
      sprintf(name, fmt[n],i+1);
      f = fopen(name, "r");
      if (!f) n++;
    } while (!f && (n<3));
    if (!f) {
      printf("loading %s  ERROR (failed to open file)\n", name);
      return -1;
    }
    sum = 0;
    do {
      l = EPROMSIZE - sum;
      r = fread(eprom[i]+sum, 1, l, f);
      if (r < 0) {
        fclose(f);
        printf("loading %s  ERROR (wrong file size: %i)\n", name, r);
        return -1;
      }
      sum += r;
    } while ((r > 0) && (sum < EPROMSIZE));
    fclose(f);
  }
  return 0;
}


int main(int argc, char *argv[])
{
  int i = 1;

  if ((argc > 1) && !strcmp(argv[i], "-s"))
  {
    i++;
    simu_code_g = 1;
  }

  if (argc != i+1)
  {
    printf("\nThis tool converts the microcode EPROM images IC1.bin, IC2.bin and IC3.bin\n");
    printf("into synthesizeable vhdl code.  Syntax: \n\n");
    printf("  %s [-s] outputfile.vhd\n\n", argv[0]);
    printf("   -s : generate vhdl code for simulation (do not add \"don't cares\")\n\n");
    return 1;
  }

  outf_g = fopen(argv[i],"w+");
  if (!outf_g)
  {
    printf("ERROR: Failed to open file %s for writing\n", argv[i]);
    return 1;
  }

  if (loadimages()<0)
    return 1;

  outputRomCode();

  free(eprom[0]);
  free(eprom[1]);
  free(eprom[2]);
  return 0;
}

