/*-----------------------------------------------------------------------------
 *
 *   Serial Interface / RS232 - Library for MS Windows and Linux
 *
 *   (c) in February 2010 by Dennis Kuschel, Germany
 *
 *   visit http://www.mycpu.eu for more information
 *
 *-----------------------------------------------------------------------------
 *
 *  Copyright (c) 2010, Dennis Kuschel.
 *  All rights reserved. 
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *   1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *   3. The name of the author may not be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission. 
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 *  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 *  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 *  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 *  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 *  OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *---------------------------------------------------------------------------*/

#include <serport.h>


#ifdef _WIN32


int serialOpen(SERPORT_t *port, const char *ttydevice, int baud, int databits, int stopbits, int parity, int handshake)
{
  SERPORT_t fd;
  COMMTIMEOUTS cto;
  DWORD mask;
  DCB   dcb;
  char comname[256];
  int i;

  if (!port)
    return -1;
  *port = NULL;

  /* create a valid device name */
  strncpy(comname, ttydevice, 255);
  i = strlen(comname);
  while ((i>0) && (comname[i-1] == ':')) i--;
  comname[i] = 0;
  for (i=0; comname[i] != 0; i++)
    comname[i] = tolower(comname[i]);
  if (sscanf(comname, "com%i", &i) != 1)
  {
    if (sscanf(comname, "%i", &i) != 1)
      return -1;
    sprintf(comname, "com%i", i);
  }

  fd = (SERPORT_t) calloc(sizeof(struct SERPORT_s), 1);
  if (!fd)
    return 0;
  
  fd->hCom = CreateFile(comname,
                        GENERIC_READ | GENERIC_WRITE,
                        0,                    // exclusive access
                        NULL,                 // no security attrs
                        OPEN_EXISTING,
                        FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED, // overlapped I/O
                        NULL );

  if ((fd->hCom==NULL) || (fd->hCom==INVALID_HANDLE_VALUE))
  {
    free(fd);
    return -1;
  }

  fd->ovlRd.hEvent = CreateEvent(NULL, TRUE, FALSE, NULL); // set to manual reset
  fd->ovlWr.hEvent = CreateEvent(NULL, TRUE, FALSE, NULL); // set to manual reset

  if ((!fd->ovlRd.hEvent) || (!fd->ovlWr.hEvent))
  {
    if (fd->ovlRd.hEvent)
      CloseHandle(fd->ovlRd.hEvent);
    if (fd->ovlWr.hEvent)
      CloseHandle(fd->ovlWr.hEvent);
    CloseHandle(fd->hCom);
    free(fd);
    return -1;
  }

  ResetEvent(fd->ovlRd.hEvent);
  ResetEvent(fd->ovlWr.hEvent);

  if (!GetCommState(fd->hCom, &dcb))
  {
    serialClose(fd);
    return -1;
  }

  dcb.BaudRate = (DWORD) baud;
  dcb.StopBits = (stopbits==1) ? ONESTOPBIT : TWOSTOPBITS;
  dcb.ByteSize = (BYTE) databits;
  dcb.Parity = (parity==0)?NOPARITY:((parity==1)?ODDPARITY:EVENPARITY);
  dcb.fBinary = 1;                        // binary mode, no EOF check
  dcb.fParity = 0;                        // enable parity checking
  dcb.fOutxCtsFlow = (handshake == 1) ? 1 : 0;   // CTS output flow control
  dcb.fOutxDsrFlow = 0;                   // DSR output flow control
  dcb.fDtrControl =  0;                   // DTR flow control type
  dcb.fDsrSensitivity = 0;                // DSR sensitivity
  dcb.fTXContinueOnXoff = 0;              // XOFF continues Tx
  dcb.fOutX = (handshake == 2) ? 1 : 0;   // XON/XOFF out flow control
  dcb.fInX =  (handshake == 2) ? 1 : 0;   // XON/XOFF in flow control
  dcb.fErrorChar = 0;                     // enable error replacement
  dcb.fNull = 0;                          // enable null stripping
  dcb.fRtsControl =                       // RTS flow control
    (handshake == 1) ? RTS_CONTROL_HANDSHAKE : RTS_CONTROL_DISABLE;
  dcb.fAbortOnError = 0;                  // abort reads/writes on error

  if (!SetCommState(fd->hCom, &dcb))
  {
    serialClose(fd);
    return -1;
  }

  if (!SetupComm(fd->hCom, 1024 /*recbuf*/, 32 /*sendbuf*/))
  {
    serialClose(fd);
    return -1;
  }

  if (!PurgeComm(fd->hCom, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR))
  {
    serialClose(fd);
    return -1;
  }

  mask = EV_RXCHAR | EV_RLSD;
  if (handshake == 1)  mask |= EV_CTS;
  if (handshake == 2)  mask |= EV_RXFLAG;

  if (!SetCommMask(fd->hCom, mask))
  {
    serialClose(fd);
    return -1;
  }

  cto.ReadIntervalTimeout = 0;
  cto.ReadTotalTimeoutMultiplier = 0;
  cto.ReadTotalTimeoutConstant = 0;
  cto.WriteTotalTimeoutMultiplier = 0;
  cto.WriteTotalTimeoutConstant = 0;

  if (!SetCommTimeouts(fd->hCom, &cto))
  {
    serialClose(fd);
    return -1;
  }

  EscapeCommFunction(fd->hCom, SETDTR);
  *port = fd;
  return 0;
}


void serialClose(SERPORT_t fd)
{
  if (fd)
  {
    if ((fd->hCom!=NULL)&&(fd->hCom!=INVALID_HANDLE_VALUE))
    {
      CloseHandle(fd->hCom);
      fd->hCom = INVALID_HANDLE_VALUE;
    }
    if (fd->ovlRd.hEvent != NULL)
    {
      CloseHandle(fd->ovlRd.hEvent);
      fd->ovlRd.hEvent = INVALID_HANDLE_VALUE;
    }
    if (fd->ovlWr.hEvent != NULL)
    {
      CloseHandle(fd->ovlWr.hEvent);
      fd->ovlWr.hEvent = INVALID_HANDLE_VALUE;
    }
    free(fd);
  }
}


int serialOutput(SERPORT_t fd, unsigned char data, unsigned long timeout)
{
  DWORD written = 0, status;

  /* adjust timeout to milliseconds */
  if (timeout && (timeout < 1000))
    timeout = 1000;
  timeout = (timeout + 500) / 1000;

  /* send data and wait until the character is out */
  fd->wrComBt = (char)data;
  ResetEvent(fd->ovlWr.hEvent);
  if (!WriteFile(fd->hCom, &fd->wrComBt, 1, &written, &fd->ovlWr))
  {
    status = GetLastError();
    if (status != 0)
    {
      if (status != ERROR_IO_PENDING)
        return -1;
      status = WaitForSingleObject(fd->ovlWr.hEvent, (DWORD)timeout);
      if (status == WAIT_TIMEOUT)
        return 0;
      if (!GetOverlappedResult(fd->hCom, &fd->ovlWr, &written, TRUE))
        return -1;
    }
  }
  return written ? 1 : 0;
}


int serialInput(SERPORT_t fd, unsigned char *pdata, unsigned timeout)
{
  DWORD read = 0, status;

  /* adjust timeout to milliseconds */
  if (timeout && (timeout < 1000))
    timeout = 1000;
  timeout = (timeout + 500) / 1000;

  /* read data from serial port */
  do
  {
    if (!fd->rdPendFlag)
    {
      ResetEvent(fd->ovlRd.hEvent);
      if (!ReadFile(fd->hCom, &fd->rdComBt, 1, &read, &fd->ovlRd))
      {
        status = GetLastError();
        if (status == ERROR_IO_PENDING)
        {
          fd->rdPendFlag = 1;
        }
        else
        {
          if (status || !read)
            return -1;
        }
      }
    }
    if (fd->rdPendFlag)
    {
      status = WaitForSingleObject(fd->ovlRd.hEvent, (DWORD)timeout);
      if (status == WAIT_TIMEOUT)
        return 0;
      if (!GetOverlappedResult(fd->hCom, &fd->ovlRd, &read, TRUE))
        read = 0;
      if (read)
        fd->rdPendFlag = 0;
    }
  }
  while (!read);
  *pdata = (unsigned char)fd->rdComBt;
  return read ? 1 : 0;
}


/*---------------------------------------------------------------------------*/
#else /* LINUX */


#define IS_TIME_AFTER(x,y)  ((((long)(y)) - ((long)(x))) < 0)

unsigned long time_getusec()
{
  static unsigned long lastus = 0;
  unsigned long usec;
  struct timeval tv;
  struct timezone tz;

  gettimeofday(&tv, &tz);

  usec = (unsigned long)(tv.tv_sec & 0xFFF) * 1000000UL;
  usec+= tv.tv_usec;
  if (usec < lastus)
    usec += 4096UL * 1000000UL;

  lastus = usec;
  return usec;
}


int serialOpen(SERPORT_t *port, const char *ttydevice, int baud, int databits, int stopbits, int parity, int handshake)
{
  unsigned long cflags, iflags;
  struct termios newtio;
  char name[256];
  int fd;
  char *dn;

  if (!port)
    return -1;

  dn = strstr(ttydevice, "dev/");
  dn = dn ? dn + 4 : (char*)ttydevice;
  snprintf(name, 255, "/dev/%s", dn);

  fd = open(name, O_RDWR|O_NOCTTY);
  if (fd < 0)
    return -1;

  cflags = CREAD | CLOCAL;
  iflags = 0;

  switch (baud)
  {
    case 300:    cflags |= B300; break;
    case 600:    cflags |= B600; break;
    case 1200:   cflags |= B1200; break;
    case 2400:   cflags |= B2400; break;
    case 4800:   cflags |= B4800; break;
    case 9600:   cflags |= B9600; break;
    case 19200:  cflags |= B19200; break;
    case 38400:  cflags |= B38400; break;
    case 57600:  cflags |= B57600; break;
    case 115200: cflags |= B115200; break;
    default:
      close(fd);
      errno = EINVAL;
      return -2;
  };

  switch (databits)
  {
    case 5: cflags |= CS5; break;
    case 6: cflags |= CS6; break;
    case 7: cflags |= CS7; break;
    case 8: cflags |= CS8; break;
    default:
      close(fd);
      errno = EINVAL;
      return -3;
  }

  switch (stopbits)
  {
    case 1: break;
    case 2: cflags |= CSTOPB; break;
    default:
      close(fd);
      errno = EINVAL;
      return -4;
  }

  switch (parity)
  {
    case 0: iflags = IGNPAR; break;
    case 1: cflags |= PARENB|PARODD; break;
    case 2: cflags |= PARENB; break;
    default:
      close(fd);
      errno = EINVAL;
      return -5;
  }

  switch (handshake)
  {
    case 0: break;
    case 1: cflags |= CRTSCTS; break;
    case 2: iflags |= IXON|IXOFF; break;
    default:
      close(fd);
      errno = EINVAL;
      return -6;
  }

  memset(&newtio, 0, sizeof(newtio));
  newtio.c_cflag = cflags;
  newtio.c_iflag = iflags;
  newtio.c_oflag = 0;
  newtio.c_lflag = 0;

  newtio.c_cc[VTIME] = 0;   /* inter-character timer unused */
  newtio.c_cc[VMIN]  = 0;   /* no blocking */

  if (iflags & (IXON|IXOFF))
  {
    newtio.c_cc[VSTART] = 17;     /* Ctrl-q */
    newtio.c_cc[VSTOP]  = 19;     /* Ctrl-s */
  }

  tcflush(fd, TCIFLUSH);
  if (tcsetattr(fd, TCSANOW, &newtio) < 0)
  {
    close(fd);
    return -7;
  }
  
  *port = fd;
  return 0;
}


void serialClose(SERPORT_t fd)
{
  close(fd);
}


int serialOutput(SERPORT_t fd, unsigned char data, unsigned long timeout)
{
  unsigned long endtime;
  int res;

  if (fd < 0)
    return -1;

  if (timeout > 0)
    endtime = time_getusec() + timeout;

  for(;;)
  {
    res = write(fd, &data, 1);
    if (res > 0)
      return 1;
    if (res < 0)
      return -1;
    if (timeout == 0)
      return 0;
    if (IS_TIME_AFTER(time_getusec(), endtime))
      return 0;
    usleep(1000);
  }
}


int serialInput(SERPORT_t fd, unsigned char *pdata, unsigned timeout)
{
  unsigned long endtime;
  unsigned long to = timeout;
  int res;

  if ((fd < 0) || !pdata)
    return -1;

  if (to > 0)
    endtime = time_getusec() + to;

  for(;;)
  {
    res = read(fd, pdata, 1);
    if (res > 0)
      return 1;
    if (res < 0)
      return -1;
    if (to == 0)
      return 0;
    if (IS_TIME_AFTER(time_getusec(), endtime))
      return 0;
    usleep(1000);
  }
}


#endif

