#include <stdio.h>
#include <string.h>
#include "serport.h"


#define ACK   0xA1
#define NACK  0xA0

struct sectordesc
{
  unsigned start;
  unsigned end;
};

#define NUM_SECTORS 11

/* bottom boot block */
struct sectordesc sectors_g[NUM_SECTORS] = {
  {0x00000, 0x03FFF},
  {0x04000, 0x05FFF},
  {0x06000, 0x07FFF},
  {0x08000, 0x0FFFF},
  {0x10000, 0x1FFFF},
  {0x20000, 0x2FFFF},
  {0x30000, 0x3FFFF},
  {0x40000, 0x4FFFF},
  {0x50000, 0x5FFFF},
  {0x60000, 0x6FFFF},
  {0x70000, 0x7FFFF}
};

static unsigned char *flashmem_g;
static int buscyc_g    = 1;
static int unlocked_g  = 0;
static int unlbypass_g = 0;
static int readinfo_g  = 0;
static int erasing_g   = 0;
static int programing_g= 0;
static int erStartAdr_g= 0;
static int erEndAdr_g  = 0;


int getSector(unsigned addr)
{
  int i;
  for (i=0; i<NUM_SECTORS; i++)
  {
    if ((addr >= sectors_g[i].start) &&
        (addr <= sectors_g[i].end))
      return i;
  }
  return 0;
}


unsigned char flash_read(unsigned addr)
{
  static unsigned char tog6 = 0;
  static unsigned char tog2 = 0;

  if (readinfo_g)
  {
    if ((addr & 0xFF) == 0x00)
    {
      return 0x01;  /* manufacturer ID */
    }
    else
    if ((addr & 0xFF) == 0x02)
    {
      return 0xBA;  /* bottom boot block */
    }
    else
    if ((addr & 0xFF) == 0x04)
    {
      return 0x00;  /* unprotected sector */
    }
    else
    {
      return 0x5A;  /* test pattern */
    }
  }

  if (erasing_g || programing_g)
  {
    programing_g = 0;
    if (erasing_g >= 8) {
      erasing_g -= 8;
    } else {
      erasing_g = 0;
    }
    if ((addr >= erStartAdr_g) &&
        (addr <= erEndAdr_g))
      tog2 = (tog2 ^ 0x04) & 0x04;
    tog6 = (tog6 ^ 0x40) & 0x40;
    return ((flashmem_g[addr] ^ 0x80) & 0x80) | tog6 | tog2 | 0x08;
  }

  return flashmem_g[addr];
}


void flash_write(unsigned addr, unsigned char data)
{
  int i;

  readinfo_g = 0;
  erasing_g = 0;
  programing_g = 0;

  switch (buscyc_g)
  {
    default:
    case 1:
    {
      if (data == 0x90)
      {
        buscyc_g = 90;
        break;
      }
      if ((data == 0xA0) && (unlbypass_g != 0))
      {
        buscyc_g = 100;
        break;
      }
      if (data == 0xF0)
      {
        unlocked_g = 0;
        unlbypass_g= 0;
        break;
      }
      if ((addr == 0xAAA) && (data == 0xAA))
      {
        buscyc_g = 2;
      }
      else
      {
        unlocked_g = 0;
        unlbypass_g= 0;
      }
      break;
    }

    case 2:
    {
      if ((addr == 0x555) && (data == 0x55))
      {
        buscyc_g = 3;
      }
      else
      {
        buscyc_g = 1;
      }
      break;
    }

    case 3:
    {
      if ((addr == 0xAAA) && (data == 0x20))
      {
        unlbypass_g = 1;
        buscyc_g = 1;
        break;
      }
      if ((addr == 0xAAA) && (data == 0x90))
      {
        readinfo_g = 1;
        break;
      }
      if ((addr == 0xAAA) && (data == 0xA0))
      {
        buscyc_g = 100;
        break;
      }
      if ((addr == 0xAAA) && (data == 0x80))
      {
        buscyc_g = 4;
        break;
      }
      buscyc_g = 1;
      break;
    }

    case 4:
    {
      if ((addr == 0xAAA) && (data == 0xAA))
      {
        buscyc_g = 5;
        break;
      }
      buscyc_g = 1;
      break;
    }

    case 5:
    {
      if ((addr == 0x555) && (data == 0x55))
      {
        buscyc_g = 6;
        break;
      }
      buscyc_g = 1;
      break;
    }

    case 6:
    {
      buscyc_g = 1;
      if ((addr == 0xAAA) && (data == 0x10))
      {
        /* chip erase */
        erasing_g = 512*1024;
        memset(flashmem_g, 0xFF, erasing_g);
        erStartAdr_g = 0;
        erEndAdr_g = 0x07FFFF;
        break;
      }
      if (data == 0x30)
      {
        /* sector erase */
        i = getSector(addr);
        printf("Erasing Sector %i, addr 0x%06x\n", i, addr);
        erStartAdr_g = sectors_g[i].start;
        erEndAdr_g   = sectors_g[i].end;
        erasing_g = erEndAdr_g - erStartAdr_g + 1;
        memset(flashmem_g + sectors_g[i].start, 0xFF, erasing_g);
        break;
      }
      break;
    }
    
    case 90:
    {
      if ((data == 0x00) || (data == 0xF0))
        unlbypass_g = 0;
      buscyc_g = 1;
      break;
    }

    case 100:
    {
      programing_g = 1;
      buscyc_g = 1;
      break;
    }
  };

  if (programing_g)
  {
     erStartAdr_g = erEndAdr_g = addr;
//printf("program 0x%06x := 0x%02x\n", addr, data);
     flashmem_g[addr] = data;
  }
}




int main(int argc, char *argv[])
{
  unsigned char *ram, rx, cmd, sendack;
  unsigned char param[4];
  SERPORT_t  port;
  unsigned addr;
  int i, n;
  
  if (argc < 2)
  {
    printf("missing parameter: serial port name\n");
    return 1;
  }

  /* open serial communication port */  
  if (serialOpen(&port, argv[1], 115200, 8, 1, 0, 0) != 0)
  {
    printf("Error: Can not open serial port '%s'\n", argv[1]);
    serialClose(port);
    return 1;
  }

  ram = calloc(1024*1024,1);
  if (!ram)
  {
    printf("Out of memory error\n");
    serialClose(port);
    return 1;
  }
  flashmem_g = malloc(512*1024);
  if (!ram)
  {
    printf("Out of memory error\n");
    free(ram);
    serialClose(port);
    return 1;
  }
  memset(flashmem_g, 0xA5, 512*1024);

  printf("MyCPU-Compact interface simulation started\n");

  for(;;)
  {
    while (!serialInput(port, &rx, 100000));
    if (rx & 0x80)  /* reset */
    {
      printf("Reset\n");
      continue;
    }

    if (rx & 0x40)
    {
      /* command interface */
      sendack = 0;
      cmd = rx;
      n = (cmd >> 4) & 3;
      for (i=0; i<n; i++)
      {
        while (!serialInput(port, &param[i], 100000));
      }
      cmd &= 0x0F;
      printf("Received command %i\n", ((int)cmd) & 15);
      sendack = NACK;
      switch (cmd)
      {
        case 0: sendack = ACK; break; /* NOP */
        case 1: if ((n == 1) && (param[0] < 3))
                {
                  switch (param[0])
                  {
                    case 0: printf("Restart CPU from Flash memory\n"); break;
                    case 1: printf("Restart CPU from RAM\n"); break;
                    case 2: printf("Let CPU run freely\n"); break;
                    default: break;
                  }
                  sendack = ACK;
                }
                break;
        case 2: if (n == 0)
                {
                  printf("Stop MyCPU\n");
                  sendack = ACK;
                }
        default: break;
      };
      if (sendack)
        serialOutput(port, sendack, 1000);
      sendack = 0;
    }
    else
    {
      /* memory interface */
      n = (rx & 0x20) ? 3 : 2;
      for (i=0; i<n; i++)
      {
        while (!serialInput(port, &param[i], 100000));
      }
      addr = ((unsigned)(rx & 0x0F)) << 16;
      addr|= ((unsigned)param[0]) << 8;
      addr|= (unsigned)param[1];
      if (rx & 0x10)
      {
        /* flash memory */
        addr &= 0x07FFFF;
        if (rx & 0x20)
        {
          flash_write(addr, param[2]);
//          printf("Write to Flash Memory, 0x%08x := 0x%02x\n", addr, ((unsigned)param[2]) & 0xFF);
        }
        else
        {
          serialOutput(port, flash_read(addr), 1000);
//          printf("Read from Flash Memory, address 0x%08x\n", addr);
        }
      }
      else
      {
        /* SRAM */
        if (rx & 0x20)
        {
          ram[addr] = param[2];
          //printf("Write to SRAM, 0x%08x := 0x%02x\n", addr, ((unsigned)param[2]) & 0xFF);
        }
        else
        {
          serialOutput(port, ram[addr], 1000);
          //printf("Read from SRAM, address 0x%08x\n", addr);
        }
      }
    }
  }

  free(ram);
  free(flashmem_g);
  serialClose(port);
  return 0;
}
