/*-----------------------------------------------------------------------------
 *
 *   Flash-ROM programming tool for MyCPU-Compact
 *
 *   (c) in February 2010 by Dennis Kuschel, Germany
 *
 *   visit http://www.mycpu.eu for more information
 *
 *-----------------------------------------------------------------------------
 *
 *  Copyright (c) 2010, Dennis Kuschel.
 *  All rights reserved. 
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *   1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *   3. The name of the author may not be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission. 
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 *  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 *  INDIRECT,  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 *  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 *  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 *  OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *---------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include "serport.h"

SERPORT_t  hPort_g;


struct sectordesc
{
  unsigned start;
  unsigned end;
};

#define RX_TIMEOUT   500000
#define TX_TIMEOUT   150000
#define NUM_SECTORS 11

struct sectordesc sectors_topboot_g[NUM_SECTORS] = {
  {0x00000, 0x0FFFF},
  {0x10000, 0x1FFFF},
  {0x20000, 0x2FFFF},
  {0x30000, 0x3FFFF},
  {0x40000, 0x4FFFF},
  {0x50000, 0x5FFFF},
  {0x60000, 0x6FFFF},
  {0x70000, 0x77FFF},
  {0x78000, 0x79FFF},
  {0x7A000, 0x7BFFF},
  {0x7C000, 0x7FFFF}
};

struct sectordesc sectors_botboot_g[NUM_SECTORS] = {
  {0x00000, 0x03FFF},
  {0x04000, 0x05FFF},
  {0x06000, 0x07FFF},
  {0x08000, 0x0FFFF},
  {0x10000, 0x1FFFF},
  {0x20000, 0x2FFFF},
  {0x30000, 0x3FFFF},
  {0x40000, 0x4FFFF},
  {0x50000, 0x5FFFF},
  {0x60000, 0x6FFFF},
  {0x70000, 0x7FFFF}
};

struct sectordesc *sectors_g = NULL;


/*---------------------------------------------------------------------------*/


/* Check if a file is available. Returns TRUE if yes.
 * path is optional and can be NULL or zero.
 * outbuf is optional and can be NULL.
 * If outbuf is given it will be filled with the name of the file.
 */
int isFileAvailable(const char *path, const char *filename, char *outbuf)
{
  char fname[512];
  FILE *f;

  *fname = 0;
  if (path)
  {
    strcpy(fname, path);
    if (*fname)
#ifdef _WIN32
      strcat(fname, "\\");
#else
      strcat(fname, "/");
#endif
  }
  strcat(fname, filename);
  f = fopen(fname, "rb");
  if (!f)
    return 0;
  fclose(f);
  if (outbuf)
    strcpy(outbuf, fname);
  return 1;
}


/*---------------------------------------------------------------------------*/


/* Establish connection to MyCPU.
 * Returns TRUE if connection was established.
 */
int connect_to_MyCPUc(void)
{
  unsigned char rx;
  int i = 0;
  int slow = 0;

  /* send some Resets */
  for (i=0; i<10; i++)
    serialOutput(hPort_g, 0x80, TX_TIMEOUT);

  /* clear the line */
  while ((i<100) && (serialInput(hPort_g, &rx, 10000) != 0)) i++;

  /* send a NOP and wait for ACK */
  if (!serialOutput(hPort_g, 0x40, TX_TIMEOUT))
    return 0;
  if (!serialInput(hPort_g, &rx, 8*RX_TIMEOUT))
    return 0;
  if (rx != 0xA1)
    return 0;

  /* send a STOP command */
  if (!serialOutput(hPort_g, 0x42, TX_TIMEOUT))
    return 0;
  if (!serialInput(hPort_g, &rx, RX_TIMEOUT/10))
  {
    slow = 1;
    if (!serialInput(hPort_g, &rx, 6*RX_TIMEOUT))
      return 0;
  }
  if (rx != 0xA1)
    return 0;

  if (slow)
    printf("Your serial port seems to be VERY SLOW. Consider to use a 'real' RS232 port.\n");

  return 1;  
}


/* does a memory write access on remote MyCPU Compact board
 */
int memWriteByte(unsigned addr, unsigned char data)
{
  unsigned char b[4];
  int i;
  b[0] = ((unsigned char) (addr >> 16) & 0x1F) | 0x20;
  b[1] = (unsigned char) (addr >> 8);
  b[2] = (unsigned char) addr;
  b[3] = data;
  for (i=0; i<4; i++)
  {
    if (!serialOutput(hPort_g, b[i], TX_TIMEOUT))
      return 0;
  }
  return 1;
}


/* does a memory read access on remote MyCPU Compact board
 */
int memReadByte(unsigned addr, unsigned char *data)
{
  unsigned char b[3];
  int i;
  b[0] = (unsigned char) (addr >> 16) & 0x1F;
  b[1] = (unsigned char) (addr >> 8);
  b[2] = (unsigned char) addr;
  for (i=0; i<3; i++)
  {
    if (!serialOutput(hPort_g, b[i], TX_TIMEOUT))
      return 0;
  }
  if (!serialInput(hPort_g, data, RX_TIMEOUT))
  {
    /* error, send reset's and try again */
    for (i=0; i<5; i++)
      serialOutput(hPort_g, 0x80, TX_TIMEOUT);
    for (i=0; i<20; i++)
      serialInput(hPort_g, data, RX_TIMEOUT/5);
    for (i=0; i<3; i++)
    {
      if (!serialOutput(hPort_g, b[i], TX_TIMEOUT))
        return 0;
    }
    if (!serialInput(hPort_g, data, RX_TIMEOUT*2))
      return 0;
  }
  return 1;
}


/* Writes a byte to Flash Memory chip.
 * Returns with "TRUE" on success.
 */
int writeFlash(unsigned addr, unsigned char data)
{
  return memWriteByte((addr & 0x07FFFF) | 0x100000, data);
}


/* Writes a byte to the SRAM Memory chips.
 * Returns with "TRUE" on success.
 */
int writeRAM(unsigned addr, unsigned char data)
{
  return memWriteByte(addr & 0x0FFFFF, data);
}


/* Reads a byte from Flash Memory chip.
 * Returns with "TRUE" on success.
 */
int readFlash(unsigned addr, unsigned char *data)
{
  return memReadByte((addr & 0x07FFFF) | 0x100000, data);
}


/* Reads a byte from the SRAM chips.
 * Returns with "TRUE" on success.
 */
int readRAM(unsigned addr, unsigned char *data)
{
  return memReadByte(addr & 0x0FFFFF, data);
}


/* Send a "run"-command to MyCPU:
 * Execute code either from ROM memory (fromRAM=0)
 * or from RAM memory (fromRAM=1)
 */
int runMyCPU(int fromRam)
{
  unsigned char rx;
  if (!serialOutput(hPort_g, 0x51, TX_TIMEOUT))
    return 0;
  if (!serialOutput(hPort_g, (fromRam != 0) ? 0x03 : 0x02 , TX_TIMEOUT))
    return 0;
  if (!serialInput(hPort_g, &rx, RX_TIMEOUT))
    return 0;
  if (rx != 0xA1)
    return 0;
  return 1;  
}


/*---------------------------------------------------------------------------*/


/* Detect flash memory chip
 * Returns 0 when error,
 *         1 when unknown flash chip,
 *         2 when S29AL004D-01 (top boot block)
 *         3 when S29AL004D-02 (bottom boot block)
 */
int detectFlash(void)
{
  unsigned char mid = 0;
  unsigned char bbtype = 0;

  sectors_g = NULL;

  if (!writeFlash(0xAAA, 0xAA))
    return 0;
  if (!writeFlash(0x555, 0x55))
    return 0;
  if (!writeFlash(0xAAA, 0x90))
    return 0;

  if (!readFlash(0x000, &mid))
    return 0;
  if (!readFlash(0x002, &bbtype))
    return 0;

  if (!writeFlash(0xAAA, 0xF0))
    return 0;

  if ((mid == 0x01) && (bbtype == 0xB9))
    return 2;  /* top boot block */
  if ((mid == 0x01) && (bbtype == 0xBA))
    return 3;  /* bottom boot block */

  return 1; /* unknown flash memory */
}


/* determine Flash memory type and setup tables
 */
int setupFlashType(void)
{
  int ft;

  if (sectors_g != NULL)
    return 1;

  ft = detectFlash();
  switch (ft)
  {
    default: return 0;
    case 2 : sectors_g = sectors_topboot_g; break;
    case 3 : sectors_g = sectors_botboot_g; break;
  }
  return 1;
}


/* erase a sector in the flash memory chip
 */
int flashEraseSector(int sector)
{
  unsigned startadr;
  unsigned char d1, d2, d3;
  int i;

  if (!setupFlashType())
    return 0;

  startadr = sectors_g[sector].start;

  /* start erase procedure */  
  if (!writeFlash(0xAAA, 0xAA))
    return 0;
  if (!writeFlash(0x555, 0x55))
    return 0;
  if (!writeFlash(0xAAA, 0x80))
    return 0;
  if (!writeFlash(0xAAA, 0xAA))
    return 0;
  if (!writeFlash(0x555, 0x55))
    return 0;
  if (!writeFlash(startadr, 0x30))
    return 0;

  /* wait until erase is finished */
  if (!readFlash(startadr, &d2))
    if (!readFlash(startadr, &d2))
      return 0;
  for (i=0; i<10; i++)
  {
    do
    {
      d1 = d2;
      if (!readFlash(startadr, &d2))
        if (!readFlash(startadr, &d2))
          return 0;
      d3 = (d1 ^ d2) & 0x44;
    }
    while (d3 != 0);
  }
  if (d2 != 0xFF)
    return 0;

  return 1;
}


/* erase a memory area in the Flash ROM
 */
int flashEraseArea(unsigned startaddr, unsigned size)
{
  unsigned adr = startaddr;
  unsigned rem = size;
  unsigned cnt;
  int s;

  /* do for all sectors of our interest */
  while (rem != 0)
  { 
    /* find next sector to erase */
    for (s=0; s<NUM_SECTORS; s++)
    {
      if ((adr >= sectors_g[s].start) &&
          (adr <= sectors_g[s].end))
        break;
    }
    cnt = sectors_g[s].end - adr + 1;
    if (rem >= cnt) {
      rem -= cnt;
    } else {
      rem = 0;
    }
    adr += cnt;
    
    /* erase sector */
    if (!flashEraseSector(s))
      return 0;
  }

  return 1;
}


/* program new data into the Flash ROM
 */
int programFlash(unsigned char *buf, unsigned size, unsigned flashaddr)
{
  unsigned char *src = buf;
  unsigned addr = flashaddr;
  unsigned rem  = size;
  unsigned a;
  unsigned char d1, d2, d3;
  int i, j, n, ret, err, prstat, prlim, proc;
  int tries;
  
  if ((flashaddr & 0xFFFF) != 0)
    return 0;   /* not at a page boundary */

  if (!setupFlashType())
  {
    printf("Error: Unknown Flash Memory Chip\n");
    return 0;
  }

  /* find first page that must be re-programmed */
  proc = 0;
  for (i=0; i<size; i++)
  {
    a = addr + i;
    if ((a & 0x3FF) == 0)
    {
      if (((a & 0xFFFF) == 0) && (i != 0))
        proc += (0x10000*100) / size;
      printf("%i%% complete  (verifying 0x%06x-0x%06x)\r", proc, a, a + 0x3FF);
      fflush(stdout);
    }
    ret = readFlash(a, &d1);
    if (!ret)
      break;
    if (d1 != src[i])
      break;
  }
  if (i < size)
    i &= 0xFF0000;
  rem  -= i;
  addr += i;
  src  += i;

  /* do for all 64k-pages */
  while (rem != 0)
  {
    tries = 3;
    prlim = ((size > 0x10000) ? 0x10000 : size) / 100;
    do
    {
      printf("%i%% complete  (erasing 0x%06x-0x%06x)    \r", proc, addr, addr + 0xFFFF);
      fflush(stdout);
      if (!flashEraseArea(addr, 0x10000))
      {
        printf("\nErase failed, programming abborted.\n");
        return 0;
      }
      
      /* write next block of data into erased page */
      n = (rem > 0x10000) ? 0x10000 : rem;
  
      ret = writeFlash(0xAAA, 0xAA);
      if (ret)
        ret = writeFlash(0x555, 0x55);
      if (ret)
        ret = writeFlash(0xAAA, 0x20);
  
      i = 0;
      a = addr;
      prstat = prlim*2;
      while (ret && (i < n))
      {
        if (prstat >= prlim)
        {
          proc = (((addr + ((i*3)/4)) - flashaddr + (prlim-1)) * 100) / size;
          if (proc > 100) proc = 100;
          printf("%i%% complete  (writing 0x%06x-0x%06x)  \r", proc, addr, addr + n - 1);
          fflush(stdout);
          prstat = 0;
        }
        prstat++;

        /* program byte */
        a = addr + i;
        if (src[i] != 0xFF)
        {
          ret = writeFlash(a, 0xA0);
          if (ret)
            ret = writeFlash(a, src[i]);
          if (!ret)
            break;
            
          /* wait until Flash is ready again */
          ret = readFlash(a, &d2);
          if (!ret)
            break;
          if ((d2 & 0x80) != (src[i] & 0x80))
          {
            j = 100;
            do
            {
              d1 = d2;
              ret = readFlash(a, &d2);
              if (!ret)
                break;
              d3 = (d1 ^ d2) & 0x44;
              j--;
            }
            while (((d3 != 0) || ((d2 & 0x80) != (src[i] & 0x80))) && (j > 0));
            if (!j)
              ret = 0;
          }
        }
  
        /* next byte */      
        i++;
      }
  
      /* stop write access */
      writeFlash(0xAAA, 0xF0);
  
      if (!ret)
      {
        printf("\nProgramming failed at address 0x%06x, programming abborted.\n", a);
        return 0;
      }
  
      /* verify programmed data */
      err = 0;
      a = addr;
      prstat = prlim*2;
      for (i=0; i<n; i++)
      {
        if (prstat >= prlim)
        {
          proc = (((addr + (i/4) + ((n*3)/4)) - flashaddr + (prlim-1)) * 100) / size;
          if (proc > 100) proc = 100;
          printf("%i%% complete  (verifying 0x%06x-0x%06x)\r", proc, addr, addr + n - 1);
          fflush(stdout);
          prstat = 0;
        }
        prstat++;

        a = addr + i;
        ret = readFlash(a, &d1);
        if (!ret)
          break;
        if (d1 != src[i])
        {
          printf("\nVerify Error at address 0x%06x\n", a);
          err = 1;
          break;
        }
      }
      if (!ret)
      {
        printf("\nVerify failed at address 0x%06x, programming abborted.\n", a);
        return 0;
      }

      /* loop (program again) when verify failed */
      tries--;
      if ((tries > 0) && err)
        printf("Retrying to program the failing sector\n");
    }
    while ((tries > 0) && err);
    
    if (err)
    {
      printf("Too many errors, giving up.\n");
      return 0;
    }


    /* next sector */
    rem  -= n;
    addr += n;
    src  += n;
  }

  printf("Memory range 0x%06x-0x%06x successfully programmed.\n", flashaddr, flashaddr+size-1);
  return 1;
}


/*---------------------------------------------------------------------------*/


/* do a quick check of the SRAM memory
 */
int checkRAM(void)
{
  unsigned a, chkaddr[22];
  unsigned char chkdata[22];
  unsigned char b, c;
  int i, err = 0;

  chkaddr[0] = 0;
  chkdata[0] = 0;
  a = 1;
  for (i=0; i<20; i++)
  {
    chkaddr[i+1] = a;
    chkdata[i+1] = (unsigned char)(((i+1)*11) ^ 0x93);
    a <<= 1;
  }
  chkaddr[21] = 0x0FFFFF;
  chkdata[21] = 0xAA;
  
  /* write data */
  for (i=0; i<22; i++)
  {
    if (!writeRAM(chkaddr[i], chkdata[i]))
      return 0;
  }
  /* read data back and compare */
  for (i=0; i<22; i++)
  {
    if (!readRAM(chkaddr[i], &b))
      return 0;
    if (chkdata[i] != b)
    {
      err++;
      printf("RAM error, failed to read address 0x%08x. Data should: 0x%02x, is: 0x%02x\n",
           chkaddr[i], ((unsigned)chkdata[i] & 0xFF), ((unsigned)b) & 0xFF);
    }
  }
  /* write data */
  for (i=0; i<22; i++)
  {
    chkdata[i] ^= 0xFF;
    if (!writeRAM(chkaddr[i], chkdata[i]))
      return 0;
  }
  /* read data back and compare */
  for (i=0; i<22; i++)
  {
    if (!readRAM(chkaddr[i], &b))
      return 0;
    if (chkdata[i] != b)
    {
      err++;
      printf("RAM error, failed to read address 0x%08x. Data should: 0x%02x, is: 0x%02x\n",
           chkaddr[i], ((unsigned)chkdata[i] & 0xFF), ((unsigned)b) & 0xFF);
    }
  }
  /* write data */
  for (i=0; i<256; i++)
  {
    if (!writeRAM((unsigned)i, (unsigned char)(i ^ 0xFF)))
      return 0;
  }
  /* read data back and compare */
  for (i=0; i<256; i++)
  {
    c = (unsigned char)(i ^ 0xFF); 
    if (!readRAM((unsigned)i, &b))
      return 0;
    if (b != c)
    {
      err++;
      printf("RAM error, failed to read address 0x%08x. Data should: 0x%02x, is: 0x%02x\n",
             i, (((unsigned)c) & 0xFF), ((unsigned)b) & 0xFF);
    }
  }

  if (err)
  {
    printf("RAM memory test:  %i errors\n", err);
  }
  else
  {
    printf("RAM memory test:  SUCCESS\n");
  }
  return 1;
}


/* do a quick check on the flash memory
 */
int checkFlash(void)
{
  int rc;

  rc = detectFlash();
  switch (rc)
  {
    default:  return 0;
    case 1: printf("Flash-ROM test :  unknown flash device (ERROR)\n"); break;
    case 2:
    case 3: printf("Flash-ROM test :  SUCCESS\n"); break;
  }
  return 1;
}


/*---------------------------------------------------------------------------*/


/* tool function: load ROM image files into a buffer
 */
unsigned char* loadFiles(const char *filename1, const char *filename2)
{
  FILE *file1, *file2;
  unsigned char *buf;
  int ofs2, size2;
  int rc;

  file1 = fopen(filename1, "rb");
  if (!file1)
  {
    printf("Failed to open file '%s'\n", filename1);
    return NULL;
  }
  
  file2 = fopen(filename2, "rb");
  if (!file2)
  {
    fclose(file1);
    printf("Failed to open file '%s'\n", filename2);
    return NULL;
  }

  buf = calloc(257*1024, 1);
  if (!buf)
  {
    fclose(file1);
    fclose(file2);
    printf("Out of memory error\n");
    return NULL;
  }

  ofs2 = 128*1024;
  size2= 32*1024;
  rc = fread(buf, 1, 128*1024, file1);
  fclose(file1);
  if ((rc != 128*1024) && (rc != 32*1024))
  {
    printf("Failed to read file '%s', wrong file size\n", filename1);
    fclose(file2);
    free(buf);
    return NULL;
  }

  /* check loaded file and swap if required */
  if (rc == size2)
  {
    memcpy(buf+ofs2, buf, size2);
    size2 = ofs2;
    ofs2  = 0;
  }

  rc = fread(buf+ofs2, 1, size2, file2);
  fgetc(file2);
  if ((rc != size2) || !feof(file2))
  {
    printf("Failed to read file '%s', wrong file size\n", filename2);
    fclose(file2);
    free(buf);
    return NULL;
  }

  fclose(file2);
  return buf;
}


/* write Operating System image files into the Flash ROM
 */
int writeOsImageToFlash(const char *filename1, const char *filename2)
{
  unsigned char *flashbuf;
  int ret;
  
  flashbuf = loadFiles(filename1, filename2);
  if (!flashbuf)
    return 0;

  printf("\nProgramming Flash Memory, please wait...\n");
  ret = programFlash(flashbuf, (128+32)*1024, 0x000000);
  
  free(flashbuf);
  return ret;
}


/* write Operating System image files into the SRAM
 */
int writeOsImageToRAM(const char *filename1, const char *filename2)
{
  #define RAMIDX(i)  (((i)<0x8000) ? (0x20000+(i)) : ((i)-0x8000))
  unsigned char *buf, d;
  int ret=1, b, i;
  int addr, a=0;
  int proc, loop, tries;
  
  buf = loadFiles(filename1, filename2);
  if (!buf)
    return 0;

  printf("\nTransfering files into SRAM, please wait...\n");
  for (b=0; b<640; b++)
  {
    addr = b * 256;

    if ((b & 3) == 0)
    {
      proc = (b*66)/640;
      printf("%i%% complete  (writing 0x%06x-0x%06x)\r", proc, addr + 0xD8000, addr + 0xD83FF);
      fflush(stdout);
    }
    tries = 5;
    do
    {
      /* write block */
      for (i=0; i<256; i++)
      {
        a = addr + i;
        ret = writeRAM(0xD8000+a, buf[RAMIDX(a)]);
        if (!ret)
        {
          printf("\nCommunication problem with MyCPU, transfer abborted.\n");
          free(buf);
          return 0;
        }
      }
      /* verify block */
      loop = 0;
      for (i=0; i<256; i++)
      {
        a = addr + i;
        ret = readRAM(0xD8000+a, &d);
        if (!ret)
        {
          printf("\nCommunication problem with MyCPU, transfer abborted.\n");
          free(buf);
          return 0;
        }
        if (d != buf[RAMIDX(a)])
        {
          loop = 1;
          tries--;
          break;
        }
      }
    }
    while (loop && (tries > 0));
    if (!tries)
    {
      printf("\nSRAM error, transfer abborted.\n");
      free(buf);
      return 0;
    }
  }
  /* verify data */
  for (addr=0; addr<0x28000; addr++)
  {
    if ((addr & 0x03FF) == 0)
    {
      proc = 66+(((addr/256)*33)/640);
      printf("%i%% complete  (verifying 0x%06x-0x%06x)\r", proc, addr, addr + 0x03FF);
      fflush(stdout);
    }
    ret = readRAM(0xD8000+addr, &d);
    if (!ret || (d != buf[RAMIDX(addr)]))
    {
      ret = readRAM(0xD8000+addr, &d);
      if (!ret)
      {
        printf("\nCommunication problem with MyCPU, transfer abborted.\n");
        free(buf);
        return 0;
      }
    }
    if (d != buf[RAMIDX(addr)])
    {
      printf("\nVerify error, transfer failed.\n");
      free(buf);
      return 0;
    }
  }
  
  printf("Transfer finished.                            \n");
  free(buf);
  return 1;
}


/*---------------------------------------------------------------------------*/


/* download memory from MyCPU-Compact
 */
int downloadMemory(unsigned char *buf, unsigned startaddr, unsigned size, int ramflag)
{
  unsigned a, addr;
  unsigned char d;
  int b, blocks, proc;
  int i, rc, err, tries;

  if (size & 0xFF)
  {
    printf("Error: 'size' is not a multiple of 256\n");
    return 0;
  }
  
  printf("\nLoading memory content from MyCPU-Compact, please wait...\n");

  /* download all blocks */
  blocks = size / 256;
  for (b=0; b<blocks; b++)
  {
    addr = b * 256;
    proc = (addr * 100) / size;
    printf("%i%% complete  (loading 0x%06x-0x%06x)\r",
           proc, startaddr+addr, startaddr+addr + 0xFF);
    fflush(stdout);

    do
    {    
      tries = 3;
      err = 0;
      /* download */
      for (i=0; i<256; i++)
      {
        a = startaddr + addr + i;
        if (ramflag) {
          rc = readRAM(a, &d);
        } else {
          rc = readFlash(a, &d);
        }
        if (!rc)
        {
          printf("\nCommunication problem with MyCPU, transfer abborted.\n");
          return 0;
        }
        buf[addr+i] = d;
      }
      /* verify */
      for (i=0; i<256; i++)
      {
        a = startaddr + addr + i;
        if (ramflag) {
          rc = readRAM(a, &d);
        } else {
          rc = readFlash(a, &d);
        }
        if (!rc)
        {
          printf("\nCommunication problem with MyCPU, transfer abborted.\n");
          return 0;
        }
        if (buf[addr+i] != d)
        {
          err = 1;
          break;
        }
      }
      tries--;
    }
    while (err && (tries > 0));
    if (err)
    {
      printf("\nCommunication problem with MyCPU, transfer abborted.\n");
      return 0;
    }
  }

  printf("Data successfully downloaded.                 \n");
  return 1;
}


/* read content of Flash ROM and store it in files
 */
int backupFlashROM(const char *filename1, const char *filename2)
{
  unsigned char *buf;
  FILE *f1, *f2;
  int rc, w;

  f1 = fopen(filename1, "w+b");
  if (!f1)
  {
    printf("Failed to open file '%s' for writing.\n", filename1);
    return 1;
  }
  f2 = fopen(filename2, "w+b");
  if (!f2)
  {
    printf("Failed to open file '%s' for writing.\n", filename2);
    fclose(f1);
    return 1;
  }

  buf = malloc(0x28000);
  if (!buf)
  {
    printf("Out of memory error.\n");
    fclose(f1);
    fclose(f2);
    return 0;
  }

  rc = downloadMemory(buf, 0x000000, 0x28000, 0);
  if (rc)
  {
    w = fwrite(buf+0x20000, 1, 0x8000, f1);
    if (w != 0x8000)
      printf("Failed to write file '%s'\n", filename1);
    w = fwrite(buf, 1, 0x20000, f2);
    if (w != 0x20000)
      printf("Failed to write file '%s'\n", filename2);
  }

  free(buf);
  fclose(f1);
  fclose(f2);
  return rc;
}


/* read OS-Image from RAM and store it in files
 */
int backupRAM(const char *filename1, const char *filename2)
{
  unsigned char *buf;
  FILE *f1, *f2;
  int rc, w;

  f1 = fopen(filename1, "w+b");
  if (!f1)
  {
    printf("Failed to open file '%s' for writing.\n", filename1);
    return 1;
  }
  f2 = fopen(filename2, "w+b");
  if (!f2)
  {
    printf("Failed to open file '%s' for writing.\n", filename2);
    fclose(f1);
    return 1;
  }

  buf = malloc(0x28000);
  if (!buf)
  {
    printf("Out of memory error.\n");
    fclose(f1);
    fclose(f2);
    return 0;
  }

  rc = downloadMemory(buf, 0xD8000, 0x28000, 1);
  if (rc)
  {
    w = fwrite(buf, 1, 0x8000, f1);
    if (w != 0x8000)
      printf("Failed to write file '%s'\n", filename1);
    w = fwrite(buf+0x8000, 1, 0x20000, f2);
    if (w != 0x20000)
      printf("Failed to write file '%s'\n", filename2);
  }

  free(buf);
  fclose(f1);
  fclose(f2);
  return rc;
}


/*---------------------------------------------------------------------------*/


int printHelp(char *progname)
{
  int i;
  i = strlen(progname);
  while ((i>0) && ((progname[i-1] != '/') && (progname[i-1] != '\\'))) i--;
  progname += i;

  printf("\nMyCPU-Compact memory transfer and test tool\n");
  printf("(c) in 2010 by Dennis Kuschel, www.mycpu.eu\n");
  printf("\nSyntax:\n");
  printf("  %s comport options [filenames]\n", progname);
  printf("\ncomport:\n");
  printf("  serial communication port, eg. com1 or /dev/ttyS0\n");
  printf("\noptions:\n");
  printf("  -f   program Operating System files into Flash-ROM\n");
  printf("  -m   transfer Operating System files into RAM memory\n");
  printf("  -b   backup content of Flash-ROM, save it into two image files\n");
  printf("  -d   download OS Image from RAM, save it into two image files\n");
  printf("  -r   run Operating System from RAM\n");
  printf("  -t   test RAM memory and Flash memory on MyCPU-Compact board\n");
  printf("\nfilenames:\n");
  printf("       Name of (ROM-)image files. Default names are IC14.bin and IC15.bin\n");
  printf("\nExamples:\n");
  printf("  %s com1 -f IC14.bin IC15.bin     -- programs OS images into the Flash\n", progname);
  printf("  %s com1 -m -r IC14.bin IC15.bin  -- loads OS to RAM and starts MyCPU\n", progname);
  printf("  %s com1 -t                       -- does a quick memory test\n", progname); 
  return 0;
}


int main(int argc, char *argv[])
{
  char filename1[512];
  char filename2[512];
  char *comport;
  int o_runRAM= 0;  // option: run Operating System from RAM
  int o_runROM= 0;  // option: run Operating System from ROM
  int o_flash = 0;  // option: copy files into Flash ROM
  int o_mem   = 0;  // option: copy files into RAM memory
  int o_test  = 0;  // option: test memory chips
  int o_rdrom = 0;  // option: download Flash memory
  int o_rdram = 0;  // option: download OS from RAM memory
  int i, fn=0;

  filename1[0] = 0;  
  filename2[0] = 0;  

  if (argc < 3)
    return printHelp(argv[0]);

  comport = argv[1];

  for (i=2; i<argc; i++)
  {
    if (argv[i][0] == '-')
    {
      if (strlen(argv[i]) != 2)
      {
        printf("Error in parameter '%s'\n", argv[i]);
        return 1;
      }
      switch (tolower(argv[i][1]))
      {
        case '?':  return printHelp(argv[0]);
        case 'r':  o_runRAM= 1; break;
        case 'f':  o_flash = 1; break;
        case 'm':  o_mem   = 1; break;
        case 't':  o_test  = 1; break;
        case 'b':  o_rdrom = 1; break;
        case 'd':  o_rdram = 1; break;
        case 'o':  o_runROM= 1; break;
        default:
          printf("Error in parameter '%s'\n", argv[i]);
          return 1;
      }
    }
    else
    {
      if (fn >= 2)
      {
        printf("Error in parameter list. Enter ' %s -? ' for help\n", argv[0]);
        return 1;
      }
      if (!fn) {
        strncpy(filename1, argv[i], 250);
      } else {
        strncpy(filename2, argv[i], 250);
      }
      fn++;
    }
  }

  /* check combination of flags */
  if (o_runRAM && o_flash)
  {
    printf("The combination of -f and -r is not supported\n");
    return 1;
  }

  /* open serial communication port */  
  if (serialOpen(&hPort_g, comport, 115200, 8, 1, 0, 0) != 0)
  {
    printf("Error: Can not open serial port '%s'\n", comport);
    return 1;
  }
  if (!connect_to_MyCPUc())
  {
    printf("Error: Can not connect to MyCPU-Compact. Ensure the serial connection\n");
    printf("       is established correctly and DIP-switch 7 is switched on.\n");
    serialClose(hPort_g);
    return 1;
  }  
  printf("Connected to MyCPU-Compact.\n");

  if (o_test)
  {
    i = checkRAM();
    i&= checkFlash();
    if (!i)
    {
      printf("Error: Communication problem with MyCPU Compact\n");
      serialClose(hPort_g);
      return 1;
    }
  }
  
  if (o_rdrom || o_rdram)
  {
    if (!*filename1 || !*filename2)
    {
      printf("Error: Missing filename in command line\n");
      serialClose(hPort_g);
      return 1;
    }
    if (isFileAvailable(NULL, filename1, NULL) ||
        isFileAvailable(NULL, filename2, NULL))
    {
      printf("Please check filenames. Files may already exist.\n");
      serialClose(hPort_g);
      return 1;
    }
  }
  if (o_rdrom)
  {
    backupFlashROM(filename1, filename2);
    serialClose(hPort_g);
    return 0;
  }
  if (o_rdram)
  {
    backupRAM(filename1, filename2);
    serialClose(hPort_g);
    return 0;
  }

  if (o_flash || o_mem)
  {
    /* find ROM image files */
    if (fn == 0)
    {
      if (!isFileAvailable(NULL, "IC14.bin", filename1))
        if (!isFileAvailable("bin", "IC14.bin", filename1))
          if (!isFileAvailable(NULL, "rom-lo.bin", filename1))
            isFileAvailable("bin", "rom-lo.bin", filename1);
      if (!isFileAvailable(NULL, "IC15.bin", filename2))
        if (!isFileAvailable("bin", "IC15.bin", filename2))
          if (!isFileAvailable(NULL, "rom-hi.bin", filename2))
            isFileAvailable("bin", "rom-hi.bin", filename2);
    }
    if (!*filename1 || !*filename2)
    {
      printf("Error: Missing filename in command line\n");
      serialClose(hPort_g);
      return 1;
    }
  }
  if (o_flash)
  {
    if (!writeOsImageToFlash(filename1, filename2))
    {
      serialClose(hPort_g);
      return 1;
    }
  }
  if (o_mem)
  {
    if (!writeOsImageToRAM(filename1, filename2))
    {
      serialClose(hPort_g);
      return 1;
    }
  }

  /* start MyCPU when requested by user */
  if (o_runRAM | o_runROM)
  {
    char *mname = o_runRAM ? "RAM" : "ROM";
    if (!runMyCPU(o_runRAM))
    {
      printf("\nError: Failed to start MyCPU from %s\n", mname);
      serialClose(hPort_g);
      return 1;
    }
    printf("MyCPU is now running the OS from %s.\n", mname);
  }
  
  serialClose(hPort_g);
  return 0;
}
